/*
 * Decompiled with CFR 0.152.
 */
package brut.androlib;

import brut.androlib.ApktoolProperties;
import brut.androlib.Config;
import brut.androlib.exceptions.AndrolibException;
import brut.androlib.exceptions.InFileNotFoundException;
import brut.androlib.exceptions.OutDirExistsException;
import brut.androlib.exceptions.UndefinedResObjectException;
import brut.androlib.meta.MetaInfo;
import brut.androlib.meta.PackageInfo;
import brut.androlib.meta.UsesFramework;
import brut.androlib.meta.VersionInfo;
import brut.androlib.res.AndrolibResources;
import brut.androlib.res.data.ResPackage;
import brut.androlib.res.data.ResTable;
import brut.androlib.res.data.ResUnknownFiles;
import brut.androlib.res.xml.ResXmlPatcher;
import brut.androlib.src.SmaliDecoder;
import brut.common.BrutException;
import brut.directory.Directory;
import brut.directory.DirectoryException;
import brut.directory.ExtFile;
import brut.util.OS;
import com.android.tools.smali.dexlib2.iface.DexFile;
import com.google.common.base.Strings;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.apache.commons.io.FilenameUtils;

public class ApkDecoder {
    private static final Logger LOGGER = Logger.getLogger(ApkDecoder.class.getName());
    private final Config config;
    private final AndrolibResources mAndRes;
    private final ExtFile mApkFile;
    private ResTable mResTable;
    protected final ResUnknownFiles mResUnknownFiles;
    private Collection<String> mUncompressedFiles;
    private int mMinSdkVersion = 0;
    private static final String[] APK_RESOURCES_FILENAMES = new String[]{"resources.arsc", "AndroidManifest.xml", "res", "r", "R"};
    private static final String[] APK_MANIFEST_FILENAMES = new String[]{"AndroidManifest.xml"};
    private static final String[] APK_STANDARD_ALL_FILENAMES = new String[]{"classes.dex", "AndroidManifest.xml", "resources.arsc", "res", "r", "R", "lib", "libs", "assets", "META-INF", "kotlin"};
    private static final Pattern NO_COMPRESS_PATTERN = Pattern.compile("(jpg|jpeg|png|gif|wav|mp2|mp3|ogg|aac|mpg|mpeg|mid|midi|smf|jet|rtttl|imy|xmf|mp4|m4a|m4v|3gp|3gpp|3g2|3gpp2|amr|awb|wma|wmv|webm|webp|mkv)$");

    public ApkDecoder(Config config, ExtFile apkFile) {
        this.config = config;
        this.mAndRes = new AndrolibResources(config);
        this.mResUnknownFiles = new ResUnknownFiles();
        this.mApkFile = apkFile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void decode(File outDir) throws AndrolibException, IOException, DirectoryException {
        try {
            if (!this.config.forceDelete && outDir.exists()) {
                throw new OutDirExistsException();
            }
            if (!this.mApkFile.isFile() || !this.mApkFile.canRead()) {
                throw new InFileNotFoundException();
            }
            try {
                OS.rmdir(outDir);
            }
            catch (BrutException ex) {
                throw new AndrolibException(ex);
            }
            outDir.mkdirs();
            LOGGER.info("Using Apktool " + ApktoolProperties.getVersion() + " on " + this.mApkFile.getName());
            if (this.hasResources()) {
                switch (this.config.decodeResources) {
                    case 256: {
                        this.decodeResourcesRaw(this.mApkFile, outDir);
                        if (this.config.forceDecodeManifest != 1 || !this.hasManifest()) break;
                        this.decodeManifestWithResources(this.mApkFile, outDir, this.getResTable());
                        break;
                    }
                    case 257: {
                        if (this.hasManifest()) {
                            this.decodeManifestWithResources(this.mApkFile, outDir, this.getResTable());
                        }
                        this.decodeResourcesFull(this.mApkFile, outDir, this.getResTable());
                    }
                }
            } else if (this.hasManifest()) {
                if (this.config.decodeResources == 257 || this.config.forceDecodeManifest == 1) {
                    this.decodeManifestFull(this.mApkFile, outDir, this.getResTable());
                } else {
                    this.decodeManifestRaw(this.mApkFile, outDir);
                }
            }
            if (this.hasSources()) {
                switch (this.config.decodeSources) {
                    case 0: {
                        this.decodeSourcesRaw(this.mApkFile, outDir, "classes.dex");
                        break;
                    }
                    case 1: 
                    case 16: {
                        this.decodeSourcesSmali(this.mApkFile, outDir, "classes.dex");
                    }
                }
            }
            if (this.hasMultipleSources()) {
                Set<String> files = this.mApkFile.getDirectory().getFiles(true);
                for (String file : files) {
                    if (!file.endsWith(".dex") || file.equalsIgnoreCase("classes.dex")) continue;
                    switch (this.config.decodeSources) {
                        case 0: {
                            this.decodeSourcesRaw(this.mApkFile, outDir, file);
                            break;
                        }
                        case 1: {
                            this.decodeSourcesSmali(this.mApkFile, outDir, file);
                            break;
                        }
                        case 16: {
                            if (file.startsWith("classes") && file.endsWith(".dex")) {
                                this.decodeSourcesSmali(this.mApkFile, outDir, file);
                                break;
                            }
                            this.decodeSourcesRaw(this.mApkFile, outDir, file);
                        }
                    }
                }
            }
            this.decodeRawFiles(this.mApkFile, outDir);
            this.decodeUnknownFiles(this.mApkFile, outDir);
            this.mUncompressedFiles = new ArrayList<String>();
            this.recordUncompressedFiles(this.mApkFile, this.mUncompressedFiles);
            this.writeOriginalFiles(this.mApkFile, outDir);
            this.writeMetaFile(outDir);
        }
        finally {
            try {
                this.mApkFile.close();
            }
            catch (IOException iOException) {}
        }
    }

    public ResTable getResTable() throws AndrolibException {
        if (this.mResTable == null) {
            boolean hasResources = this.hasResources();
            boolean hasManifest = this.hasManifest();
            if (!hasManifest && !hasResources) {
                throw new AndrolibException("Apk doesn't contain either AndroidManifest.xml file or resources.arsc file");
            }
            this.mResTable = this.mAndRes.getResTable(this.mApkFile, hasResources);
            this.mResTable.setAnalysisMode(this.config.analysisMode);
        }
        return this.mResTable;
    }

    public boolean hasSources() throws AndrolibException {
        try {
            return this.mApkFile.getDirectory().containsFile("classes.dex");
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public boolean hasMultipleSources() throws AndrolibException {
        try {
            Set<String> files = this.mApkFile.getDirectory().getFiles(false);
            for (String file : files) {
                if (!file.endsWith(".dex") || file.equalsIgnoreCase("classes.dex")) continue;
                return true;
            }
            return false;
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public boolean hasManifest() throws AndrolibException {
        try {
            return this.mApkFile.getDirectory().containsFile("AndroidManifest.xml");
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public boolean hasResources() throws AndrolibException {
        try {
            return this.mApkFile.getDirectory().containsFile("resources.arsc");
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void close() throws IOException {
        this.mAndRes.close();
    }

    public void writeMetaFile(File outDir) throws AndrolibException {
        MetaInfo meta = new MetaInfo();
        meta.version = ApktoolProperties.getVersion();
        meta.apkFileName = this.mApkFile.getName();
        if (this.mResTable != null) {
            meta.isFrameworkApk = this.mResTable.isFrameworkApk();
            this.putUsesFramework(meta);
            this.putSdkInfo(outDir, meta);
            this.putPackageInfo(meta);
            this.putVersionInfo(outDir, meta);
            this.putSharedLibraryInfo(meta);
            this.putSparseResourcesInfo(meta);
        } else {
            this.putMinSdkInfo(meta);
        }
        this.putUnknownInfo(meta);
        this.putFileCompressionInfo(meta);
        try {
            meta.save(new File(outDir, "apktool.yml"));
        }
        catch (IOException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void putUsesFramework(MetaInfo meta) {
        Set<ResPackage> pkgs = this.mResTable.listFramePackages();
        if (pkgs.isEmpty()) {
            return;
        }
        Object[] ids = new Integer[pkgs.size()];
        int i = 0;
        for (ResPackage pkg : pkgs) {
            ids[i++] = pkg.getId();
        }
        Arrays.sort(ids);
        meta.usesFramework = new UsesFramework();
        meta.usesFramework.ids = Arrays.asList(ids);
        if (this.config.frameworkTag != null) {
            meta.usesFramework.tag = this.config.frameworkTag;
        }
    }

    private void putSdkInfo(File outDir, MetaInfo meta) {
        Map<String, String> info = this.mResTable.getSdkInfo();
        if (info.size() > 0) {
            String refValue;
            if (info.get("minSdkVersion") != null && (refValue = ResXmlPatcher.pullValueFromIntegers(outDir, info.get("minSdkVersion"))) != null) {
                info.put("minSdkVersion", refValue);
            }
            if (info.get("targetSdkVersion") != null && (refValue = ResXmlPatcher.pullValueFromIntegers(outDir, info.get("targetSdkVersion"))) != null) {
                info.put("targetSdkVersion", refValue);
            }
            if (info.get("maxSdkVersion") != null && (refValue = ResXmlPatcher.pullValueFromIntegers(outDir, info.get("maxSdkVersion"))) != null) {
                info.put("maxSdkVersion", refValue);
            }
            meta.sdkInfo = info;
        }
    }

    private void putMinSdkInfo(MetaInfo meta) {
        if (this.mMinSdkVersion > 0) {
            LinkedHashMap<String, String> sdkInfo = new LinkedHashMap<String, String>();
            sdkInfo.put("minSdkVersion", Integer.toString(this.mMinSdkVersion));
            meta.sdkInfo = sdkInfo;
        }
    }

    private void putPackageInfo(MetaInfo meta) throws AndrolibException {
        String renamed = this.mResTable.getPackageRenamed();
        String original = this.mResTable.getPackageOriginal();
        int id = this.mResTable.getPackageId();
        try {
            id = this.mResTable.getPackage(renamed).getId();
        }
        catch (UndefinedResObjectException undefinedResObjectException) {
            // empty catch block
        }
        if (Strings.isNullOrEmpty(original)) {
            return;
        }
        meta.packageInfo = new PackageInfo();
        if (!renamed.equalsIgnoreCase(original)) {
            meta.packageInfo.renameManifestPackage = renamed;
        }
        meta.packageInfo.forcedPackageId = String.valueOf(id);
    }

    private void putVersionInfo(File outDir, MetaInfo meta) {
        VersionInfo info = this.mResTable.getVersionInfo();
        String refValue = ResXmlPatcher.pullValueFromStrings(outDir, info.versionName);
        if (refValue != null) {
            info.versionName = refValue;
        }
        meta.versionInfo = info;
    }

    private void putSharedLibraryInfo(MetaInfo meta) {
        meta.sharedLibrary = this.mResTable.getSharedLibrary();
    }

    private void putSparseResourcesInfo(MetaInfo meta) {
        meta.sparseResources = this.mResTable.getSparseResources();
    }

    private void putUnknownInfo(MetaInfo meta) {
        meta.unknownFiles = this.mResUnknownFiles.getUnknownFiles();
    }

    private void putFileCompressionInfo(MetaInfo meta) {
        if (this.mUncompressedFiles != null && !this.mUncompressedFiles.isEmpty()) {
            meta.doNotCompress = this.mUncompressedFiles;
        }
    }

    public void decodeSourcesRaw(ExtFile apkFile, File outDir, String filename) throws AndrolibException {
        try {
            LOGGER.info("Copying raw " + filename + " file...");
            apkFile.getDirectory().copyToDir(outDir, filename);
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void decodeSourcesSmali(File apkFile, File outDir, String filename) throws AndrolibException {
        try {
            File smaliDir = filename.equalsIgnoreCase("classes.dex") ? new File(outDir, "smali") : new File(outDir, "smali_" + filename.substring(0, filename.indexOf(".")));
            OS.rmdir(smaliDir);
            smaliDir.mkdirs();
            LOGGER.info("Baksmaling " + filename + "...");
            DexFile dexFile = SmaliDecoder.decode(apkFile, smaliDir, filename, this.config.baksmaliDebugMode, this.config.apiLevel);
            int minSdkVersion = dexFile.getOpcodes().api;
            if (this.mMinSdkVersion == 0 || this.mMinSdkVersion > minSdkVersion) {
                this.mMinSdkVersion = minSdkVersion;
            }
        }
        catch (BrutException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void decodeManifestRaw(ExtFile apkFile, File outDir) throws AndrolibException {
        try {
            LOGGER.info("Copying raw manifest...");
            apkFile.getDirectory().copyToDir(outDir, APK_MANIFEST_FILENAMES);
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void decodeManifestFull(ExtFile apkFile, File outDir, ResTable resTable) throws AndrolibException {
        this.mAndRes.decodeManifest(resTable, apkFile, outDir);
    }

    public void decodeResourcesRaw(ExtFile apkFile, File outDir) throws AndrolibException {
        try {
            LOGGER.info("Copying raw resources...");
            apkFile.getDirectory().copyToDir(outDir, APK_RESOURCES_FILENAMES);
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void decodeResourcesFull(ExtFile apkFile, File outDir, ResTable resTable) throws AndrolibException {
        this.mAndRes.decode(resTable, apkFile, outDir);
    }

    public void decodeManifestWithResources(ExtFile apkFile, File outDir, ResTable resTable) throws AndrolibException {
        this.mAndRes.decodeManifestWithResources(resTable, apkFile, outDir);
    }

    public void decodeRawFiles(ExtFile apkFile, File outDir) throws AndrolibException {
        LOGGER.info("Copying assets and libs...");
        try {
            Directory in = apkFile.getDirectory();
            if (this.config.decodeAssets == 1 && in.containsDir("assets")) {
                in.copyToDir(outDir, "assets");
            }
            if (in.containsDir("lib")) {
                in.copyToDir(outDir, "lib");
            }
            if (in.containsDir("libs")) {
                in.copyToDir(outDir, "libs");
            }
            if (in.containsDir("kotlin")) {
                in.copyToDir(outDir, "kotlin");
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private boolean isAPKFileNames(String file) {
        for (String apkFile : APK_STANDARD_ALL_FILENAMES) {
            if (!apkFile.equals(file) && !file.startsWith(apkFile + "/")) continue;
            return true;
        }
        return false;
    }

    public void decodeUnknownFiles(ExtFile apkFile, File outDir) throws AndrolibException {
        LOGGER.info("Copying unknown files...");
        File unknownOut = new File(outDir, "unknown");
        try {
            Directory unk = apkFile.getDirectory();
            Set<String> files = unk.getFiles(true);
            for (String file : files) {
                if (this.isAPKFileNames(file) || file.endsWith(".dex")) continue;
                unk.copyToDir(unknownOut, file);
                this.mResUnknownFiles.addUnknownFileInfo(file, String.valueOf(unk.getCompressionLevel(file)));
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void writeOriginalFiles(ExtFile apkFile, File outDir) throws AndrolibException {
        LOGGER.info("Copying original files...");
        File originalDir = new File(outDir, "original");
        if (!originalDir.exists()) {
            originalDir.mkdirs();
        }
        try {
            Directory in = apkFile.getDirectory();
            if (in.containsFile("AndroidManifest.xml")) {
                in.copyToDir(originalDir, "AndroidManifest.xml");
            }
            if (in.containsFile("stamp-cert-sha256")) {
                in.copyToDir(originalDir, "stamp-cert-sha256");
            }
            if (in.containsDir("META-INF")) {
                in.copyToDir(originalDir, "META-INF");
                if (in.containsDir("META-INF/services")) {
                    LOGGER.info("Copying META-INF/services directory");
                    in.copyToDir(outDir, "META-INF/services");
                }
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    public void recordUncompressedFiles(ExtFile apkFile, Collection<String> uncompressedFilesOrExts) throws AndrolibException {
        try {
            Directory unk = apkFile.getDirectory();
            Set<String> files = unk.getFiles(true);
            for (String file : files) {
                if (!this.isAPKFileNames(file) || unk.getCompressionLevel(file) != 0) continue;
                String extOrFile = "";
                if (unk.getSize(file) != 0L) {
                    extOrFile = FilenameUtils.getExtension(file);
                }
                if ((extOrFile.isEmpty() || !NO_COMPRESS_PATTERN.matcher(extOrFile).find()) && this.mAndRes.mResFileMapping.containsKey(extOrFile = file)) {
                    extOrFile = this.mAndRes.mResFileMapping.get(extOrFile);
                }
                if (uncompressedFilesOrExts.contains(extOrFile)) continue;
                uncompressedFilesOrExts.add(extOrFile);
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }
}

