import functions as fn


class Packages:
    def __init__(self):
        self.packages_file_name = "packages-x86_64.txt"
        self.default_export_path = "{}/{}".format(fn.home, self.packages_file_name)
        self.process_timeout = 20
        self.install_process_timeout = 600  # 10m timeout on pacman package install
        self.start_logtimer = False
        # Create a queue for storing Pacman log file contents
        self.pacmanlog_queue = fn.Queue()
        # Create a queue for storing Packages status
        self.packages_status_queue = fn.Queue()
        # Create a queue for storing Package install failures messages
        self.packages_err_queue = fn.Queue()
        # Log file to store package install status
        self.logfile = "%spackages-install-status-%s-%s.log" % (
            fn.att_log_dir,
            fn.datetime.datetime.today().date(),
            fn.datetime.datetime.today().time().strftime("%H-%M-%S"),
        )

    # export package list to file
    def export_packages(self, export_selected):
        try:
            # pacman -Qqen - for packages from pacman repos native and does not capture AUR
            # pacman -Qqem - for packages from the AUR
            if export_selected == "export_all":
                query_str = ["pacman", "-Qq"]
            if export_selected == "export_explicit":
                query_str = ["pacman", "-Qqen"]

            if self.pacman_sync() is False:
                return

            fn.logger.info("Running %s" % " ".join(query_str))

            process_pkg_query = fn.subprocess.Popen(
                query_str,
                shell=False,
                stdout=fn.subprocess.PIPE,
                stderr=fn.subprocess.PIPE,
            )

            out, err = process_pkg_query.communicate(timeout=self.process_timeout)

            if process_pkg_query.returncode == 0:
                with open(self.default_export_path, "w") as f:
                    f.write(
                        "# This file was auto-generated by the ArchLinux Tweak Tool on %s at %s\n"
                        % (
                            fn.datetime.datetime.today().date(),
                            fn.datetime.datetime.now().strftime("%H:%M:%S"),
                        )
                    )
                    if export_selected == "export_all":
                        f.write(
                            "# Exported all installed packages using %s\n"
                            % " ".join(query_str)
                        )
                    if export_selected == "export_explicit":
                        f.write(
                            "# Exported explicitly installed packages using %s\n"
                            % " ".join(query_str)
                        )
                    for line in out.decode("utf-8").splitlines():
                        f.write("%s\n" % line)

                if fn.os.path.exists(self.default_export_path):
                    lines = []
                    with open(self.default_export_path, "r") as r:
                        lines = r.readlines()
                    if len(lines) > 1:
                        fn.permissions(self.default_export_path)
                        lines.clear()
                        return True
                    else:
                        return False
                else:
                    return False
            else:
                return False

        except Exception as e:
            fn.logger.error("Exception in export_packages(): %s" % e)
            return False

    # install packages
    def install_packages(
        self,
        packages,
        button_install,
        gui_parts,
    ):
        try:
            fn.logger.debug(
                "Attempting to read in packages file from %s" % self.default_export_path
            )
            if packages is not None:
                thread_addlog = "thread_addPacmanLogQueue"
                thread_add_pacmanlog_alive = fn.is_thread_alive(thread_addlog)

                # spawn a new thread to keep monitoring the pacman log for new content, and add to pacman_log queue
                if thread_add_pacmanlog_alive is False:
                    fn.logger.info("Starting monitor Pacman Log file thread")
                    th_add_pacmanlog_queue = fn.threading.Thread(
                        name=thread_addlog,
                        target=fn.add_pacmanlog_queue,
                        args=(self,),
                        daemon=True,
                    )
                    th_add_pacmanlog_queue.start()

                    fn.logger.info("Thread monitor Pacman Log file started")
                else:
                    fn.logger.info("Thread monitor Pacman Log file is already alive")

                thread_logtimer = "thread_startLogTimer"
                thread_logtimer_alive = fn.is_thread_alive(thread_logtimer)

                self.start_logtimer = True

                if thread_logtimer_alive is False:
                    fn.logger.info("Starting Log timer thread")
                    th_logtimer = fn.threading.Thread(
                        name=thread_logtimer,
                        target=fn.start_log_timer,
                        args=(self, gui_parts[3], gui_parts[4]),
                        daemon=True,
                    )
                    th_logtimer.start()

                    fn.logger.debug("Thread Log timer started")
                else:
                    fn.logger.debug("Thread Log timer is already alive")

                thread_install_packages = fn.threading.Thread(
                    target=self.pacman_install_packages,
                    daemon=True,
                    args=(gui_parts[1], packages, button_install, gui_parts[5]),
                )
                thread_install_packages.start()

                thread_check_install_packages = fn.threading.Thread(
                    target=self.log_package_status, daemon=True
                )
                thread_check_install_packages.start()

            else:
                fn.logger.error(
                    "Not a valid packages file, the ArchLinux Tweak Tool header comment was not found"
                )

        except Exception as e:
            fn.logger.error("Exception in install_packages(): %s" % e)

    # package install completed now log status to log file
    def log_package_status(self):
        packages_status_list = None
        package_err = None
        while True:
            try:
                fn.time.sleep(0.2)
                packages_status_list = self.packages_status_queue.get()
                package_err = self.packages_err_queue.get()

            finally:
                self.packages_status_queue.task_done()
                self.packages_err_queue.task_done()
                with open(self.logfile, "w") as f:
                    f.write(
                        "# This file was auto-generated by the ArchLinux Tweak Tool on %s at %s\n"
                        % (
                            fn.datetime.datetime.today().date(),
                            fn.datetime.datetime.now().strftime("%H:%M:%S"),
                        ),
                    )
                    if packages_status_list is not None:
                        for package in packages_status_list:
                            if package.split("->")[0].strip() in package_err:
                                f.write("%s\n" % package)
                                f.write(
                                    "\tERROR: %s\n"
                                    % package_err[package.split("->")[0].strip()]
                                )
                            else:
                                f.write("%s\n" % package)

                break

    # pacman synchronize database
    def pacman_sync(self):
        query_pacman_sync_str = ["pacman", "-Sy"]

        process_pacman_sync = fn.subprocess.Popen(
            query_pacman_sync_str,
            shell=False,
            stdout=fn.subprocess.PIPE,
            stderr=fn.subprocess.PIPE,
        )

        out, err = process_pacman_sync.communicate(timeout=self.process_timeout)

        if process_pacman_sync.returncode == 0:
            fn.logger.info("Pacman synchronizing package lists completed")
        else:
            fn.logger.error("Pacman synchronizing package lists failed")
            return

    # this is run inside another thread
    def pacman_install_packages(
        self, label_install_status, packages, button, label_install_count
    ):
        try:
            fn.logger.info("Installing packages")
            packages_status_list = []
            package_failed = False
            package_err = {}

            count = 0

            # run pacman sync before running install

            if self.pacman_sync() is False:
                return

            # clean pacman cache

            if fn.os.path.exists(fn.pacman_cache_dir):
                query_pacman_clean_cache_str = ["pacman", "-Sc", "--noconfirm"]

                fn.logger.info(
                    "Cleaning Pacman cache directory = %s" % fn.pacman_cache_dir
                )

                # clean the pacman cache, so we don't run into any invalid/corrupt package errors during install
                process_pacman_cc = fn.subprocess.Popen(
                    query_pacman_clean_cache_str,
                    shell=False,
                    stdout=fn.subprocess.PIPE,
                    stderr=fn.subprocess.PIPE,
                )

                out, err = process_pacman_cc.communicate(timeout=self.process_timeout)

                if process_pacman_cc.returncode == 0:
                    fn.logger.info("Pacman cache directory cleaned")
                else:
                    fn.logger.error("Failed to clean Pacman cache directory")

            fn.logger.info("Running full system upgrade")
            # run full system upgrade, Arch does not allow partial package updates
            query_str = ["pacman", "-Syu", "--noconfirm"]

            process_pacman_upgrade = fn.subprocess.Popen(
                query_str,
                shell=False,
                stdout=fn.subprocess.PIPE,
                stderr=fn.subprocess.PIPE,
            )

            fn.GLib.idle_add(
                fn.update_package_status_label,
                self,
                label_install_status,
                "Status: <b>Performing full system upgrade - do not power off your system</b>",
            )

            out, err = process_pacman_upgrade.communicate(timeout=900)

            if process_pacman_upgrade.returncode == 0:
                fn.logger.info("Pacman system upgrade completed")
                fn.GLib.idle_add(
                    fn.update_package_status_label,
                    self,
                    label_install_status,
                    "Status: <b> Full system upgrade - completed</b>",
                )
            else:
                output = out.decode("utf-8").splitlines()

                if len(output) > 0:
                    if "there is nothing to do" not in out:
                        fn.logger.error("Pacman system upgrade failed")
                        fn.GLib.idle_add(
                            fn.update_package_status_label,
                            self,
                            label_install_status,
                            "Status: <b> Full system upgrade - failed</b>",
                        )

                        fn.logger.error("%s" % " ".join(output))

                        # do not proceed with package installs if system upgrade fails
                        return

            # iterate through list of packages, calling pacman -S on each one
            for package in packages:
                package = package.strip()
                if len(package) > 0:
                    if "#" not in package:
                        query_str = ["pacman", "-S", package, "--needed", "--noconfirm"]

                        count += 1

                        fn.logger.info("Running %s" % " ".join(query_str))

                        process_pkg_query = fn.subprocess.Popen(
                            query_str,
                            shell=False,
                            stdout=fn.subprocess.PIPE,
                            stderr=fn.subprocess.PIPE,
                        )

                        out, err = process_pkg_query.communicate(
                            timeout=self.install_process_timeout
                        )

                        if process_pkg_query.returncode == 0:
                            # since this is being run in another thread outside of main, use GLib to update UI component
                            fn.GLib.idle_add(
                                fn.update_package_status_label,
                                self,
                                label_install_status,
                                "Status: <b>%s</b> -> <b>Installed</b>" % package,
                            )

                            fn.GLib.idle_add(
                                fn.update_package_status_label,
                                self,
                                label_install_count,
                                "Progress: <b>%s/%s</b>" % (count, len(packages)),
                            )

                            packages_status_list.append("%s -> Installed" % package)

                        else:
                            fn.logger.error("%s --> Install failed" % package)
                            fn.GLib.idle_add(
                                fn.update_package_status_label,
                                self,
                                label_install_status,
                                "Status: <b>%s</b> -> <b>Install failed</b>" % package,
                            )

                            fn.GLib.idle_add(
                                fn.update_package_status_label,
                                self,
                                label_install_count,
                                "Progress: <b>%s/%s</b>" % (count, len(packages)),
                            )

                            output = out.decode("utf-8").splitlines()

                            if len(output) > 0:
                                if "there is nothing to do" not in output:
                                    fn.logger.error("%s" % " ".join(output))
                                    # store package error in dict
                                    package_err[package] = " ".join(output)

                            package_failed = True

                            packages_status_list.append("%s -> Failed" % package)

            # flag to stop monitoring pacman log
            self.start_logtimer = False

            if len(packages_status_list) > 0:
                self.packages_status_queue.put(packages_status_list)

            if package_failed is True:
                fn.GLib.idle_add(
                    fn.update_package_status_label,
                    self,
                    label_install_status,
                    "<b>Some packages have failed to install see %s</b>" % self.logfile,
                )

            button.set_sensitive(True)

        except Exception as e:
            fn.logger.error("Exception in pacman_install_packages(): %s" % e)
            button.set_sensitive(True)
            # flag to stop monitoring pacman log
            self.start_logtimer = False
        finally:
            self.packages_err_queue.put(package_err)

    # read package file contents into memory
    def get_packages_file_content(self):
        try:
            if fn.os.path.exists(self.default_export_path):
                lines = []
                with open(self.default_export_path, "r") as r:
                    lines = r.readlines()

                # the line "# This file was auto-generated by the ArchLinux Tweak Tool on should be present in the package list file
                # this is a form of check that we have a valid file
                if len(lines) > 1:
                    if (
                        "# This file was auto-generated by the ArchLinux Tweak Tool on"
                        in lines[0]
                    ):
                        packages_list = []
                        for package in lines:
                            if len(package.strip()) and "#" not in package.strip():
                                packages_list.append(package.strip())

                        return packages_list

                    else:
                        return None
                else:
                    return None
            else:
                return None
        except Exception as e:
            fn.logger.error("Exception in get_packages_file_content(): %s" % e)
            return None
