/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.cpachecker.util.pixelexport;

import com.google.common.base.Preconditions;
import com.google.common.primitives.ImmutableIntArray;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayDeque;
import java.util.HashSet;
import javax.imageio.ImageIO;
import javax.imageio.stream.FileImageOutputStream;
import org.apache.batik.dom.GenericDOMImplementation;
import org.apache.batik.svggen.SVGGraphics2D;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.configuration.Option;
import org.sosy_lab.common.configuration.Options;
import org.sosy_lab.cpachecker.util.Pair;
import org.sosy_lab.cpachecker.util.pixelexport.GraphLevel;
import org.sosy_lab.cpachecker.util.pixelexport.GraphStructure;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;

@Options(prefix="pixelgraphic.export")
public abstract class GraphToPixelsWriter<Node> {
    @Option(secure=true, description="Padding of the bitmap on the left and right (each) in pixels")
    private int xPadding = 2;
    @Option(secure=true, description="Padding of the bitmap on the top and bottom (each) in pixels")
    private int yPadding = 2;
    @Option(secure=true, description="Width of the bitmap in pixels. If set to -1, width is computed in relation to the height. If both are set to -1, the optimal bitmap size to represent the graph is used. The final width is width*scaling")
    private int width = -1;
    @Option(secure=true, description="Height of the bitmap in pixels. If set to -1, height is  computed in relation to the width. If both are set to -1, the optimal bitmap size to represent the graph is used. The final height is height*scaling")
    private int height = -1;
    @Option(secure=true, description="Scaling of the bitmap. If set to 1, 1 pixel represents one graph node. If set to 2, 2 * 2 pixels represent one graph node, and so on.")
    private int scaling = 2;
    @Option(secure=true, description="Format to use for image output", name="format")
    private String imageFormat = "svg";
    @Option(secure=true, description="Highlight not only corresponding graph nodes, but background of corresponding line, too. This may give an better overview, but also introduces more clutter")
    private boolean strongHighlight = true;
    public static final Color COLOR_BACKGROUND = Color.LIGHT_GRAY;
    public static final Color COLOR_NODE = Color.BLACK;
    private static final String FORMAT_SVG = "svg";
    private final CanvasProvider canvasHandler;

    protected GraphToPixelsWriter(Configuration pConfig) throws InvalidConfigurationException {
        pConfig.inject((Object)this, GraphToPixelsWriter.class);
        this.imageFormat = this.imageFormat.toLowerCase();
        if (this.width == 0 || this.height == 0) {
            throw new InvalidConfigurationException("Width and height may not be 0");
        }
        if (this.scaling == 0) {
            throw new InvalidConfigurationException("Scaling may not be 0");
        }
        this.canvasHandler = this.imageFormat.equals(FORMAT_SVG) ? new SvgProvider() : new BitmapProvider(this.imageFormat);
    }

    public GraphStructure getStructure(Node pRoot) {
        GraphStructure structure = new GraphStructure();
        ArrayDeque<Pair<Integer, Object>> worklist = new ArrayDeque<Pair<Integer, Object>>();
        HashSet reached = new HashSet();
        worklist.add(Pair.of(0, pRoot));
        int oldDistance = 0;
        GraphLevel.Builder<Node> levelBuilder = this.getLevelBuilder();
        while (!worklist.isEmpty()) {
            Pair current = (Pair)Preconditions.checkNotNull((Object)((Pair)worklist.poll()));
            if (reached.contains(current.getSecond())) continue;
            reached.add(current.getSecond());
            int currentDistance = (Integer)Preconditions.checkNotNull((Object)((Integer)current.getFirst()));
            if (oldDistance != currentDistance) {
                structure.addLevel(levelBuilder.build());
                oldDistance = currentDistance;
                levelBuilder = this.getLevelBuilder();
            }
            Object currentNode = Preconditions.checkNotNull(current.getSecond());
            levelBuilder.node();
            levelBuilder.addMarkings(currentNode);
            for (Object s : this.getChildren(currentNode)) {
                if (reached.contains(s)) continue;
                worklist.add(Pair.of(currentDistance + 1, s));
            }
        }
        structure.addLevel(levelBuilder.build());
        return structure;
    }

    public abstract GraphLevel.Builder<Node> getLevelBuilder();

    public abstract Iterable<Node> getChildren(Node var1);

    private int getWidth(GraphStructure pGraphStructure) throws InvalidConfigurationException {
        int finalWidth;
        int neededWidth = this.scaling * pGraphStructure.getMaxWidth() + this.xPadding * 2;
        int intendedWidth = this.scaling * this.width;
        if (intendedWidth > 0) {
            if (intendedWidth < neededWidth) {
                throw new InvalidConfigurationException("Graph doesn't fit on the defined canvas. Needed width: " + neededWidth);
            }
            finalWidth = intendedWidth;
        } else {
            finalWidth = neededWidth;
        }
        return finalWidth;
    }

    private int getHeight(GraphStructure pArgStructure) throws InvalidConfigurationException {
        int finalHeight;
        int neededHeight = this.scaling * pArgStructure.getDepth() + this.yPadding * 2;
        int intendedHeight = this.scaling * this.height;
        if (intendedHeight > 0) {
            if (intendedHeight < neededHeight) {
                throw new InvalidConfigurationException("Graph doesn't fit on the defined canvas. Needed height: " + neededHeight);
            }
            finalHeight = intendedHeight;
        } else {
            finalHeight = neededHeight;
        }
        return finalHeight;
    }

    private void drawContent(Graphics2D pCanvas, int pWidth, int pHeight, GraphStructure pGraphStructure) {
        pCanvas.setColor(COLOR_BACKGROUND);
        pCanvas.fillRect(0, 0, pWidth, pHeight);
        int middle = pWidth / 2;
        int yPos = this.yPadding;
        for (GraphLevel level : pGraphStructure) {
            int stateNum = level.getWidth();
            int lineWidth = stateNum * this.scaling;
            int xPos = middle - lineWidth / 2;
            if (this.strongHighlight) {
                Color levelBackground = level.getBackgroundColor();
                pCanvas.setColor(levelBackground);
                pCanvas.fillRect(0, yPos, pWidth, this.scaling);
            }
            pCanvas.setColor(COLOR_NODE);
            pCanvas.fillRect(xPos, yPos, lineWidth, this.scaling);
            int currentYPos = yPos;
            for (Pair<ImmutableIntArray, Color> p : level.getGroups()) {
                pCanvas.setColor(p.getSecondNotNull());
                p.getFirstNotNull().forEach(idx -> pCanvas.fillRect(xPos + (idx - 1) * this.scaling, currentYPos, this.scaling, this.scaling));
            }
            yPos += this.scaling;
        }
    }

    public void write(Node pRoot, Path pOutputFile) throws IOException, InvalidConfigurationException {
        GraphStructure structure = this.getStructure(pRoot);
        int finalWidth = this.getWidth(structure);
        int finalHeight = this.getHeight(structure);
        Graphics2D g = this.canvasHandler.createCanvas(finalWidth, finalHeight);
        this.drawContent(g, finalWidth, finalHeight, structure);
        Path fullOutputFile = Path.of(pOutputFile + "." + this.imageFormat, new String[0]);
        this.canvasHandler.writeToFile(fullOutputFile);
    }

    private static class SvgProvider
    implements CanvasProvider {
        private SVGGraphics2D svgGenerator = null;

        private SvgProvider() {
        }

        @Override
        public Graphics2D createCanvas(int pWidth, int pHeight) {
            DOMImplementation domImpl = GenericDOMImplementation.getDOMImplementation();
            String svgNS = "http://www.w3.org/2000/svg";
            Document document = domImpl.createDocument(svgNS, GraphToPixelsWriter.FORMAT_SVG, null);
            this.svgGenerator = new SVGGraphics2D(document);
            this.svgGenerator.setSVGCanvasSize(new Dimension(pWidth, pHeight));
            return this.svgGenerator;
        }

        @Override
        public void writeToFile(Path pOutputFile) throws IOException {
            Preconditions.checkState((this.svgGenerator != null ? 1 : 0) != 0, (Object)"Canvas not created");
            try (BufferedWriter out = Files.newBufferedWriter(pOutputFile, Charset.defaultCharset(), new OpenOption[0]);){
                this.svgGenerator.stream((Writer)out);
            }
        }
    }

    private static class BitmapProvider
    implements CanvasProvider {
        private String imageFormat;
        private BufferedImage bufferedImage = null;

        BitmapProvider(String pFormat) {
            this.imageFormat = pFormat;
        }

        @Override
        public Graphics2D createCanvas(int pWidth, int pHeight) {
            Preconditions.checkState((this.bufferedImage == null ? 1 : 0) != 0, (Object)"createCanvas can only be called after writing the old canvas to a file");
            this.bufferedImage = new BufferedImage(pWidth, pHeight, 5);
            return this.bufferedImage.createGraphics();
        }

        @Override
        public void writeToFile(Path pOutputFile) throws IOException, InvalidConfigurationException {
            Preconditions.checkState((this.bufferedImage != null ? 1 : 0) != 0, (Object)"Canvas not created");
            try (FileImageOutputStream out = new FileImageOutputStream(pOutputFile.toFile());){
                boolean success = ImageIO.write((RenderedImage)this.bufferedImage, this.imageFormat, out);
                if (!success) {
                    throw new InvalidConfigurationException("ImageIO can't handle given format: " + this.imageFormat);
                }
            }
            this.bufferedImage = null;
        }
    }

    private static interface CanvasProvider {
        public Graphics2D createCanvas(int var1, int var2);

        public void writeToFile(Path var1) throws IOException, InvalidConfigurationException;
    }
}

