/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.cpachecker.util.resources;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.math.LongMath;
import com.google.common.primitives.Longs;
import java.math.RoundingMode;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import javax.management.JMException;
import org.sosy_lab.common.Concurrency;
import org.sosy_lab.common.ShutdownManager;
import org.sosy_lab.common.ShutdownNotifier;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.configuration.Option;
import org.sosy_lab.common.configuration.Options;
import org.sosy_lab.common.configuration.TimeSpanOption;
import org.sosy_lab.common.log.LogManager;
import org.sosy_lab.common.time.TimeSpan;
import org.sosy_lab.cpachecker.util.resources.ProcessCpuTimeLimit;
import org.sosy_lab.cpachecker.util.resources.ResourceLimit;
import org.sosy_lab.cpachecker.util.resources.ThreadCpuTimeLimit;
import org.sosy_lab.cpachecker.util.resources.WalltimeLimit;

public final class ResourceLimitChecker {
    private final Thread thread;
    private final List<ResourceLimit> limits;

    public ResourceLimitChecker(ShutdownManager shutdownManager, List<ResourceLimit> limits) {
        Preconditions.checkNotNull((Object)shutdownManager);
        this.limits = limits;
        this.thread = limits.isEmpty() || shutdownManager.getNotifier().shouldShutdown() ? null : Concurrency.newDaemonThread((String)"Resource limit checker", (Runnable)new ResourceLimitCheckRunnable(shutdownManager, limits));
    }

    public void start() {
        if (this.thread != null) {
            this.thread.start();
        }
    }

    public void cancel() {
        if (this.thread != null) {
            this.thread.interrupt();
        }
    }

    public List<ResourceLimit> getResourceLimits() {
        return this.limits;
    }

    public static ResourceLimitChecker fromConfiguration(Configuration config, LogManager logger, ShutdownManager shutdownManager) throws InvalidConfigurationException {
        ImmutableList limitsList;
        boolean cpuTimeLimitSet;
        ResourceLimitOptions options = new ResourceLimitOptions();
        config.inject((Object)options);
        ImmutableList.Builder limits = ImmutableList.builder();
        if (options.walltime.compareTo(TimeSpan.empty()) >= 0) {
            limits.add((Object)WalltimeLimit.fromNowOn(options.walltime));
        }
        boolean bl = cpuTimeLimitSet = options.cpuTime.compareTo(TimeSpan.empty()) >= 0;
        if (options.cpuTimeRequired.compareTo(TimeSpan.empty()) >= 0 && !options.cpuTimeRequired.equals((Object)options.cpuTime)) {
            if (!cpuTimeLimitSet) {
                throw new InvalidConfigurationException("CPU time limit was not specified but is required to be explicitly set to " + options.cpuTimeRequired + " in this configuration.");
            }
            throw new InvalidConfigurationException("CPU time limit was set to " + options.cpuTime + "  but is required to be explicitly set to " + options.cpuTimeRequired + " in this configuration.");
        }
        if (cpuTimeLimitSet) {
            try {
                limits.add((Object)ProcessCpuTimeLimit.fromNowOn(options.cpuTime));
            }
            catch (JMException e) {
                logger.logDebugException((Throwable)e, "Querying cpu time failed");
                logger.log(Level.WARNING, new Object[]{"Your Java VM does not support measuring the cpu time, cpu time threshold disabled."});
            }
        }
        if (options.threadTime.compareTo(TimeSpan.empty()) >= 0) {
            limits.add((Object)ThreadCpuTimeLimit.fromNowOn(options.threadTime, Thread.currentThread()));
        }
        if (!(limitsList = limits.build()).isEmpty()) {
            logger.log(Level.INFO, new Object[]{"Using the following resource limits:", Joiner.on((String)", ").join((Iterable)Lists.transform((List)limitsList, ResourceLimit::getName))});
        }
        return new ResourceLimitChecker(shutdownManager, (List<ResourceLimit>)limitsList);
    }

    public static ResourceLimitChecker createCpuTimeLimitChecker(LogManager logger, ShutdownManager shutdownManager, TimeSpan cpuTime) {
        if (cpuTime.compareTo(TimeSpan.empty()) <= 0) {
            return new ResourceLimitChecker(shutdownManager, (List<ResourceLimit>)ImmutableList.of());
        }
        try {
            ProcessCpuTimeLimit cpuTimeLimitChecker = ProcessCpuTimeLimit.fromNowOn(cpuTime);
            logger.log(Level.INFO, new Object[]{"Using " + cpuTimeLimitChecker.getName()});
            return new ResourceLimitChecker(shutdownManager, (List<ResourceLimit>)ImmutableList.of((Object)cpuTimeLimitChecker));
        }
        catch (JMException e) {
            logger.log(Level.WARNING, new Object[]{"Your Java VM does not support measuring the cpu time, cpu time threshold disabled."});
            return new ResourceLimitChecker(shutdownManager, (List<ResourceLimit>)ImmutableList.of());
        }
    }

    private static class ResourceLimitCheckRunnable
    implements Runnable {
        private static final long PRECISION = TimeUnit.MILLISECONDS.toNanos(500L);
        private final ShutdownManager shutdownManager;
        private final ImmutableList<ResourceLimit> limits;

        ResourceLimitCheckRunnable(ShutdownManager pShutdownManager, List<ResourceLimit> pLimits) {
            this.shutdownManager = (ShutdownManager)Preconditions.checkNotNull((Object)pShutdownManager);
            this.limits = ImmutableList.copyOf(pLimits);
            Preconditions.checkArgument((!this.limits.isEmpty() ? 1 : 0) != 0);
        }

        @Override
        public void run() {
            ShutdownNotifier.ShutdownRequestListener interruptThreadOnShutdown = ShutdownNotifier.interruptCurrentThreadOnShutdown();
            this.shutdownManager.getNotifier().registerAndCheckImmediately(interruptThreadOnShutdown);
            long[] timesOfNextCheck = new long[this.limits.size()];
            while (true) {
                long currentTime = System.nanoTime();
                int i = 0;
                for (ResourceLimit limit : this.limits) {
                    if (currentTime < timesOfNextCheck[i]) {
                        ++i;
                        continue;
                    }
                    long currentValue = limit.getCurrentValue();
                    if (limit.isExceeded(currentValue)) {
                        this.updateCurrentValuesOfAllLimits();
                        String reason = String.format("The %s has elapsed.", limit.getName());
                        this.shutdownManager.requestShutdown(reason);
                        return;
                    }
                    long nanosToNextCheck = limit.nanoSecondsToNextCheck(currentValue);
                    timesOfNextCheck[i] = currentTime + nanosToNextCheck;
                    ++i;
                }
                long timeOfNextCheck = Longs.min((long[])timesOfNextCheck);
                long nanosToSleep = Math.max(timeOfNextCheck - currentTime, PRECISION);
                long millisToSleep = LongMath.divide((long)nanosToSleep, (long)1000000L, (RoundingMode)RoundingMode.UP);
                try {
                    Thread.sleep(millisToSleep);
                }
                catch (InterruptedException e) {
                    this.updateCurrentValuesOfAllLimits();
                    this.shutdownManager.getNotifier().unregister(interruptThreadOnShutdown);
                    return;
                }
            }
        }

        private void updateCurrentValuesOfAllLimits() {
            for (ResourceLimit l : this.limits) {
                l.getCurrentValue();
            }
        }
    }

    @Options(prefix="limits")
    private static class ResourceLimitOptions {
        @Option(secure=true, name="time.wall", description="Limit for wall time used by CPAchecker (use seconds or specify a unit; -1 for infinite)")
        @TimeSpanOption(codeUnit=TimeUnit.NANOSECONDS, defaultUserUnit=TimeUnit.SECONDS, min=-1L)
        private TimeSpan walltime = TimeSpan.ofNanos((long)-1L);
        @Option(secure=true, name="time.cpu", description="Limit for cpu time used by CPAchecker (use seconds or specify a unit; -1 for infinite)")
        @TimeSpanOption(codeUnit=TimeUnit.NANOSECONDS, defaultUserUnit=TimeUnit.SECONDS, min=-1L)
        private TimeSpan cpuTime = TimeSpan.ofNanos((long)-1L);
        @Option(secure=true, name="time.cpu::required", description="Enforce that the given CPU time limit is set as the value of limits.time.cpu.")
        @TimeSpanOption(codeUnit=TimeUnit.NANOSECONDS, defaultUserUnit=TimeUnit.SECONDS, min=-1L)
        private TimeSpan cpuTimeRequired = TimeSpan.ofNanos((long)-1L);
        @Option(secure=true, name="time.cpu.thread", description="Limit for thread cpu time used by CPAchecker. This option will in general not work when multi-threading is used in more than one place, use only with great caution! (use seconds or specify a unit; -1 for infinite)")
        @TimeSpanOption(codeUnit=TimeUnit.NANOSECONDS, defaultUserUnit=TimeUnit.SECONDS, min=-1L)
        private TimeSpan threadTime = TimeSpan.ofNanos((long)-1L);

        private ResourceLimitOptions() {
        }
    }
}

