import collections
import os
import queue
import sqlite3
import threading
import traceback
import time

from hydrus.core import HydrusDBBase
from hydrus.core import HydrusConstants as HC
from hydrus.core import HydrusData
from hydrus.core import HydrusEncryption
from hydrus.core import HydrusExceptions
from hydrus.core import HydrusGlobals as HG
from hydrus.core import HydrusPaths
from hydrus.core import HydrusProfiling
from hydrus.core import HydrusTime
from hydrus.core.interfaces import HydrusControllerInterface

def CheckCanVacuum( db_path, stop_time = None ):
    
    db = sqlite3.connect( db_path, isolation_level = None, detect_types = sqlite3.PARSE_DECLTYPES )
    
    c = db.cursor()
    
    CheckCanVacuumCursor( db_path, c, stop_time = stop_time )
    
def CheckCanVacuumCursor( db_path, c, stop_time = None ):
    
    ( page_size, ) = c.execute( 'PRAGMA page_size;' ).fetchone()
    ( page_count, ) = c.execute( 'PRAGMA page_count;' ).fetchone()
    ( freelist_count, ) = c.execute( 'PRAGMA freelist_count;' ).fetchone()
    
    CheckCanVacuumData( db_path, page_size, page_count, freelist_count, stop_time = stop_time )
    
def CheckCanVacuumData( db_path, page_size, page_count, freelist_count, stop_time = None ):
    
    db_size = ( page_count - freelist_count ) * page_size
    
    if stop_time is not None:
        
        approx_vacuum_duration = GetApproxVacuumDuration( db_size )
        
        time_i_will_have_to_start = stop_time - approx_vacuum_duration
        
        if HydrusTime.TimeHasPassed( time_i_will_have_to_start ):
            
            raise Exception( 'I believe you need about ' + HydrusTime.TimeDeltaToPrettyTimeDelta( approx_vacuum_duration ) + ' to vacuum, but there is not enough time allotted.' )
            
        
    
    db_dir = os.path.dirname( db_path )
    
    HydrusDBBase.CheckHasSpaceForDBTransaction( db_dir, db_size )
    

def GetApproxVacuumDuration( db_size ):
    
    vacuum_estimate = int( db_size * 1.2 )
    
    approx_vacuum_speed_mb_per_s = 1048576 * 1
    
    approx_vacuum_duration = vacuum_estimate // approx_vacuum_speed_mb_per_s
    
    return approx_vacuum_duration
    

def ReadLargeIdQueryInSeparateChunks( cursor, select_statement, chunk_size ):
    
    table_name = 'tempbigread' + os.urandom( 32 ).hex()
    
    cursor.execute( 'CREATE TEMPORARY TABLE ' + table_name + ' ( job_id INTEGER PRIMARY KEY AUTOINCREMENT, temp_id INTEGER );' )
    
    cursor.execute( 'INSERT INTO ' + table_name + ' ( temp_id ) ' + select_statement ) # given statement should end in semicolon, so we are good
    
    num_to_do = cursor.rowcount
    
    if num_to_do is None or num_to_do == -1:
        
        num_to_do = 0
        
    
    i = 0
    num_done = 0
    
    while num_done < num_to_do:
        
        chunk = [ temp_id for ( temp_id, ) in cursor.execute( 'SELECT temp_id FROM ' + table_name + ' WHERE job_id BETWEEN ? AND ?;', ( i, i + ( chunk_size - 1 ) ) ) ]
        
        i += chunk_size
        num_done += len( chunk )
        
        yield ( chunk, num_done, num_to_do )
        
    
    cursor.execute( 'DROP TABLE ' + table_name + ';' )
    

def VacuumDB( db_path ):
    
    db = sqlite3.connect( db_path, isolation_level = None, detect_types = sqlite3.PARSE_DECLTYPES )
    
    c = db.cursor()
    
    fast_big_transaction_wal = not sqlite3.sqlite_version_info < ( 3, 11, 0 )
    
    if HG.db_journal_mode == 'WAL' and not fast_big_transaction_wal:
        
        c.execute( 'PRAGMA journal_mode = TRUNCATE;' )
        
    
    # this used to be 1024 for Linux users, so we do want to check and coerce back to SQLite default, 4096
    
    ( page_size, ) = c.execute( 'PRAGMA page_size;' ).fetchone()
    
    ideal_page_size = 4096
    
    if page_size != ideal_page_size:
        
        c.execute( 'PRAGMA journal_mode = TRUNCATE;' )
        c.execute( 'PRAGMA page_size = ' + str( ideal_page_size ) + ';' )
        
    
    c.execute( 'PRAGMA auto_vacuum = 0;' ) # none
    
    c.execute( 'VACUUM;' )
    
    c.execute( 'PRAGMA journal_mode = {};'.format( HG.db_journal_mode ) )
    

class HydrusDB( HydrusDBBase.DBBase ):
    
    READ_WRITE_ACTIONS = []
    UPDATE_WAIT = 2
    
    def __init__( self, controller: HydrusControllerInterface.HydrusControllerInterface, db_dir, db_name ):
        
        super().__init__()
        
        self._controller = controller
        self._db_dir = db_dir
        self._db_name = db_name
        
        self._modules = []
        
        HydrusDBBase.TemporaryIntegerTableNameCache()
        
        self._ssl_cert_filename = '{}.crt'.format( self._db_name )
        self._ssl_key_filename = '{}.key'.format( self._db_name )
        
        self._ssl_cert_path = os.path.join( self._db_dir, self._ssl_cert_filename )
        self._ssl_key_path = os.path.join( self._db_dir, self._ssl_key_filename )
        
        self._finished_job_event = threading.Event()
        
        main_db_filename = db_name
        
        if not main_db_filename.endswith( '.db' ):
            
            main_db_filename += '.db'
            
        
        self._db_filenames = {}
        
        self._db_filenames[ 'main' ] = main_db_filename
        
        self._durable_temp_db_filename = db_name + '.temp.db'
        
        durable_temp_db_path = os.path.join( self._db_dir, self._durable_temp_db_filename )
        
        if os.path.exists( durable_temp_db_path ):
            
            HydrusPaths.DeletePath( durable_temp_db_path )
            
            wal_lad = durable_temp_db_path + '-wal'
            
            if os.path.exists( wal_lad ):
                
                HydrusPaths.DeletePath( wal_lad )
                
            
            shm_lad = durable_temp_db_path + '-shm'
            
            if os.path.exists( shm_lad ):
                
                HydrusPaths.DeletePath( shm_lad )
                
            
            HydrusData.Print( 'Found and deleted the durable temporary database on boot. The last exit was probably not clean.' )
            
        
        self._InitExternalDatabases()
        
        self._is_first_start = False
        self._is_db_updated = False
        self._local_shutdown = False
        self._pause_and_disconnect = False
        self._loop_finished = False
        self._ready_to_serve_requests = False
        self._could_not_initialise = False
        
        self._jobs = queue.Queue()
        
        self._currently_doing_job = False
        self._current_status = ''
        self._current_job_name = ''
        
        self._db = None
        self._is_connected = False
        
        self._cursor_transaction_wrapper = None
        
        if os.path.exists( os.path.join( self._db_dir, self._db_filenames[ 'main' ] ) ):
            
            # open and close to clean up in case last session didn't close well
            
            self._InitDB()
            self._CloseDBConnection()
            
        
        total_db_size = self.GetApproxTotalFileSize()
        
        size_check = min( int( total_db_size * 0.5 ), 500 * 1048576 )
        
        size_check = max( size_check, 64 * 1048576 )
        
        free_space = HydrusPaths.GetFreeSpace( db_dir )
        
        if free_space is not None and free_space < size_check:
            
            raise HydrusExceptions.DBAccessException( 'Sorry, it looks like the database drive partition has less than {} free space. It needs this for database transactions, so please free up some space.'.format( HydrusData.ToHumanBytes( size_check ) ) )
            
        
        self._InitDB()
        
        ( version, ) = self._Execute( 'SELECT version FROM version;' ).fetchone()
        
        if version > HC.SOFTWARE_VERSION:
            
            self._ReportOverupdatedDB( version )
            
        
        if version < HC.SOFTWARE_VERSION - 50:
            
            raise HydrusExceptions.DBVersionException( 'Your current database version of hydrus ' + str( version ) + ' is too old for this software version ' + str( HC.SOFTWARE_VERSION ) + ' to update. Please try updating with version ' + str( version + 45 ) + ' or earlier first.' )
            
        
        bitrot_rows = [
            ( 'client', 551, 558, 'millisecond timestamp conversion' )
        ]
        
        for ( bitrot_db_name, latest_affected_version, safe_update_version, reason ) in bitrot_rows:
            
            if self._db_name == bitrot_db_name and version <= latest_affected_version:
                
                raise HydrusExceptions.DBVersionException( f'Sorry, due to a bitrot issue ({reason}), you cannot update to this software version (v{HC.SOFTWARE_VERSION}) if your database is on v{latest_affected_version} or earlier (you are on v{version}). Please download and update to v{safe_update_version} first!' )
                
            
        
        if version < ( HC.SOFTWARE_VERSION - 15 ):
            
            self._ReportUnderupdatedDB( version )
            
        
        self._RepairDB( version )
        
        while version < HC.SOFTWARE_VERSION:
            
            time.sleep( self.UPDATE_WAIT )
            
            try:
                
                self._cursor_transaction_wrapper.BeginImmediate()
                
            except Exception as e:
                
                raise HydrusExceptions.DBAccessException( str( e ) )
                
            
            try:
                
                self._UpdateDB( version )
                
                self._cursor_transaction_wrapper.Commit()
                
                self._is_db_updated = True
                
            except:
                
                e = Exception( 'Updating the ' + self._db_name + ' db to version ' + str( version + 1 ) + ' caused this error:' + '\n' + traceback.format_exc() )
                
                try:
                    
                    self._cursor_transaction_wrapper.Rollback()
                    
                except Exception as rollback_e:
                    
                    HydrusData.Print( 'When the update failed, attempting to rollback the database failed.' )
                    
                    HydrusData.PrintException( rollback_e )
                    
                
                raise e
                
            
            ( version, ) = self._Execute( 'SELECT version FROM version;' ).fetchone()
            
        
        self._CloseDBConnection()
        
        self._controller.CallToThreadLongRunning( self.MainLoop )
        
        while not self._ready_to_serve_requests:
            
            time.sleep( 0.1 )
            
            if self._could_not_initialise:
                
                raise Exception( 'Could not initialise the db! Error written to the log!' )
                
            
        
    
    def _AttachExternalDatabases( self ):
        
        for ( name, filename ) in self._db_filenames.items():
            
            if name == 'main':
                
                continue
                
            
            db_path = os.path.join( self._db_dir, filename )
            
            if os.path.exists( db_path ) and not HydrusPaths.FileisWriteable( db_path ):
                
                raise HydrusExceptions.DBAccessException( '"{}" seems to be read-only!'.format( db_path ) )
                
            
            self._Execute( 'ATTACH ? AS ' + name + ';', ( db_path, ) )
            
        
        db_path = os.path.join( self._db_dir, self._durable_temp_db_filename )
        
        self._Execute( 'ATTACH ? AS durable_temp;', ( db_path, ) )
        
    
    def _CleanAfterJobWork( self ):
        
        self._cursor_transaction_wrapper.CleanPubSubs()
        
    
    def _CloseDBConnection( self ):
        
        HydrusDBBase.TemporaryIntegerTableNameCache.instance().Clear()
        
        if self._db is not None:
            
            if self._cursor_transaction_wrapper.InTransaction():
                
                self._cursor_transaction_wrapper.Commit()
                
            
            self._CloseCursor()
            
            self._db.close()
            
            del self._db
            
            self._db = None
            
            self._is_connected = False
            
            self._cursor_transaction_wrapper = None
            
            self._UnloadModules()
            
        
    
    def _CreateDB( self ):
        
        raise NotImplementedError()
        
    
    def _DisplayCatastrophicError( self, text ):
        
        message = 'The db encountered a serious error! This is going to be written to the log as well, but here it is for a screenshot:'
        message += '\n' * 2
        message += text
        
        HydrusData.DebugPrint( message )
        
    
    def _DoAfterJobWork( self ):
        
        self._cursor_transaction_wrapper.DoPubSubs()
        
    
    def _GenerateDBJob( self, job_type, synchronous, action, *args, **kwargs ):
        
        return HydrusDBBase.JobDatabase( job_type, synchronous, action, *args, **kwargs )
        
    
    def _GetPossibleAdditionalDBFilenames( self ):
        
        return [ self._ssl_cert_filename, self._ssl_key_filename ]
        
    
    def _InitCaches( self ):
        
        pass
        
    
    def _InitDB( self ):
        
        main_database_is_missing = False
        
        main_db_path = os.path.join( self._db_dir, self._db_filenames[ 'main' ] )
        
        external_db_paths = [ os.path.join( self._db_dir, self._db_filenames[ db_name ] ) for db_name in self._db_filenames if db_name != 'main' ]
        
        existing_external_db_paths = [ external_db_path for external_db_path in external_db_paths if os.path.exists( external_db_path ) ]
        
        if os.path.exists( main_db_path ):
            
            if len( existing_external_db_paths ) < len( external_db_paths ):
                
                external_paths_summary = '"{}"'.format( '", "'.join( [ path for path in external_db_paths if path not in existing_external_db_paths ] ) )
                
                message = f'While the main database file, "{main_db_path}", exists, the external files {external_paths_summary} do not!\n\nIf this is a surprise to you, you have probably had a hard drive failure. You must close this process immediately and diagnose what has happened. Check the "help my db is broke.txt" document in the install_dir/db directory for additional help.\n\nIf this is not a surprise, then you may continue if you wish, and hydrus will do its best to reconstruct the missing files. You will see more error prompts.'
                
                self._controller.BlockingSafeShowCriticalMessage( 'missing database file!', message )
                
            
        else:
            
            main_database_is_missing = True
            
            if len( existing_external_db_paths ) > 0:
                
                external_paths_summary = '"{}"'.format( '", "'.join( existing_external_db_paths ) )
                
                message = f'Although the external files, {external_paths_summary} do exist, the main database file, "{main_db_path}", does not!\n\nThis makes for an invalid database, and the program will now quit. Please contact hydrus_dev if you do not know how this happened or need help recovering from hard drive failure.'
                
                raise HydrusExceptions.DBAccessException( message )
                
            
        
        self._InitDBConnection()
        
        version_is_missing = False
        
        result = self._Execute( 'SELECT 1 FROM sqlite_master WHERE type = ? AND name = ?;', ( 'table', 'version' ) ).fetchone()
        
        if result is None:
            
            if not main_database_is_missing:
                
                message = f'The "version" table in your {main_db_path} database was missing.\n\nIf you have used this database many times before, then you have probably had a hard drive failure. You must close this process immediately and diagnose what has happened. Check the "help my db is broke.txt" document in the install_dir/db directory for additional help.\n\nIf this database is new, and you recently attempted to boot it for the first time, but it failed, then this is less of a worrying situation, and you can continue.'
                
                self._controller.BlockingSafeShowCriticalMessage( 'missing critical database table!', message )
                
            
            version_is_missing = True
            
        
        if main_database_is_missing or version_is_missing:
            
            self._is_first_start = True
            
            self._CreateDB()
            
            self._cursor_transaction_wrapper.CommitAndBegin()
            
        
    
    def _InitDBConnection( self ):
        
        self._CloseDBConnection()
        
        db_path = os.path.join( self._db_dir, self._db_filenames[ 'main' ] )
        
        try:
            
            if os.path.exists( db_path ) and not HydrusPaths.FileisWriteable( db_path ):
                
                raise HydrusExceptions.DBAccessException( '"{}" seems to be read-only!'.format( db_path ) )
                
            
            self._db = sqlite3.connect( db_path, isolation_level = None, detect_types = sqlite3.PARSE_DECLTYPES )
            
            c = self._db.cursor()
            
            self._SetCursor( c )
            
            self._is_connected = True
            
            self._cursor_transaction_wrapper = HydrusDBBase.DBCursorTransactionWrapper( self._c, HG.db_transaction_commit_period )
            
            if HG.no_db_temp_files:
                
                self._Execute( 'PRAGMA temp_store = 2;' ) # use memory for temp store exclusively
                
            
            self._AttachExternalDatabases()
            
            self._LoadModules()
            
            self._Execute( 'ATTACH ":memory:" AS mem;' )
            
        except HydrusExceptions.DBAccessException as e:
            
            raise
            
        except Exception as e:
            
            raise HydrusExceptions.DBAccessException( 'Could not connect to database! If the answer is not obvious to you, please let hydrus dev know. Error follows:' + '\n' * 2 + str( e ) )
            
        
        HydrusDBBase.TemporaryIntegerTableNameCache.instance().Clear()
        
        # durable_temp is not excluded here
        db_names = [ name for ( index, name, path ) in self._Execute( 'PRAGMA database_list;' ) if name not in ( 'mem', 'temp' ) ]
        
        for db_name in db_names:
            
            # MB -> KB
            cache_size = HG.db_cache_size * 1024
            
            self._Execute( 'PRAGMA {}.cache_size = -{};'.format( db_name, cache_size ) )
            
            self._Execute( 'PRAGMA {}.journal_mode = {};'.format( db_name, HG.db_journal_mode ) )
            
            if HG.db_journal_mode in ( 'PERSIST', 'WAL' ):
                
                # We tried 1GB here, but I have reports of larger ones that don't seem to truncate ever?
                # Not sure what that is about, but I guess the db sometimes doesn't want to (expensively?) recover pages from the journal and just appends more data
                # In any case, this pragma is not a 'don't allow it to grow larger than', it's a 'after commit, truncate back to this', so no need to make it so large
                # default is -1, which means no limit
                
                self._Execute( 'PRAGMA {}.journal_size_limit = {};'.format( db_name, HydrusDBBase.JOURNAL_SIZE_LIMIT ) )
                
            
            self._Execute( 'PRAGMA {}.synchronous = {};'.format( db_name, HG.db_synchronous ) )
            
            try:
                
                self._Execute( 'SELECT * FROM {}.sqlite_master;'.format( db_name ) ).fetchone()
                
            except sqlite3.OperationalError as e:
                
                message = 'The database seemed valid, but hydrus failed to read basic data from it. You may need to run the program in a different journal mode using --db_journal_mode. Full error information:'
                
                message += '\n' * 2
                message += str( e )
                
                HydrusData.DebugPrint( message )
                
                raise HydrusExceptions.DBAccessException( message )
                
            
        
        try:
            
            self._cursor_transaction_wrapper.BeginImmediate()
            
        except Exception as e:
            
            if 'locked' in str( e ):
                
                raise HydrusExceptions.DBAccessException( 'Database appeared to be locked. Please ensure there is not another client already running on this database, and then try restarting the client.' )
                
            
            raise HydrusExceptions.DBAccessException( str( e ) )
            
        
    
    def _InitExternalDatabases( self ):
        
        pass
        
    
    def _LoadModules( self ):
        
        pass
        
    
    def _ManageDBError( self, job, e ):
        
        raise NotImplementedError()
        
    
    def _ProcessJob( self, job: HydrusDBBase.JobDatabase ):
        
        job_type = job.GetType()
        
        ( action, args, kwargs ) = job.GetCallableTuple()
        
        try:
            
            if job_type in ( 'read_write', 'write' ):
                
                self._current_status = 'db write locked'
                
                self._cursor_transaction_wrapper.NotifyWriteOccuring()
                
            else:
                
                self._current_status = 'db read locked'
                
            
            self.publish_status_update()
            
            idle_at_job_start = self._controller.CurrentlyIdle()
            time_job_started = HydrusTime.GetNowPrecise()
            
            result = None
            
            if job_type in ( 'read', 'read_write' ):
                
                result = self._Read( action, *args, **kwargs )
                
            elif job_type in ( 'write' ):
                
                result = self._Write( action, *args, **kwargs )
                
            
            idle_at_job_end = self._controller.CurrentlyIdle()
            
            if not idle_at_job_start or not idle_at_job_end:
                
                time_job_finished = HydrusTime.GetNowPrecise()
                
                time_job_took = time_job_finished - time_job_started
                
                if time_job_took > 15:
                    
                    HydrusData.Print( f'The database job "{job.ToString()}" took {HydrusTime.TimeDeltaToPrettyTimeDelta( time_job_took )}.' )
                    
                
            
            if job.IsSynchronous():
                
                job.PutResult( result )
                
            
            self._cursor_transaction_wrapper.Save()
            
            if self._cursor_transaction_wrapper.TimeToCommit():
                
                self._current_status = 'db committing'
                
                self.publish_status_update()
                
                self._cursor_transaction_wrapper.CommitAndBegin()
                
            
            self._DoAfterJobWork()
            
        except Exception as e:
            
            self._ManageDBError( job, e )
            
            try:
                
                self._cursor_transaction_wrapper.Rollback()
                
            except Exception as rollback_e:
                
                HydrusData.Print( 'When the transaction failed, attempting to rollback the database failed. Please restart the client as soon as is convenient.' )
                
                self._CloseDBConnection()
                
                self._InitDBConnection()
                
                HydrusData.PrintException( rollback_e )
                
            
        finally:
            
            self._CleanAfterJobWork()
            
            self._current_status = ''
            
            self.publish_status_update()
            
        
    
    def _Read( self, action, *args, **kwargs ):
        
        raise NotImplementedError()
        
    
    def _RepairDB( self, version ):
        
        for module in self._modules:
            
            module.Repair( version, self._cursor_transaction_wrapper )
            
        
        if HG.controller.LastShutdownWasBad():
            
            for module in self._modules:
                
                module.DoLastShutdownWasBadWork()
                
            
        
    
    def _ReportOverupdatedDB( self, version ):
        
        pass
        
    
    def _ReportUnderupdatedDB( self, version ):
        
        pass
        
    
    def _ShrinkMemory( self ):
        
        self._Execute( 'PRAGMA shrink_memory;' )
        
    
    def _UnloadModules( self ):
        
        self._modules = []
        
    
    def _UpdateDB( self, version ):
        
        raise NotImplementedError()
        
    
    def _Write( self, action, *args, **kwargs ):
        
        raise NotImplementedError()
        
    
    def publish_status_update( self ):
        
        pass
        
    
    def CurrentlyDoingJob( self ):
        
        return self._currently_doing_job
        
    
    def GetApproxTotalFileSize( self ):
        
        total = 0
        
        for filename in self._db_filenames.values():
            
            path = os.path.join( self._db_dir, filename )
            
            if os.path.exists( path ):
                
                total += os.path.getsize( path )
                
            
        
        return total
        
    
    def GetSSLPaths( self ):
        
        # create ssl keys
        
        cert_here = os.path.exists( self._ssl_cert_path )
        key_here = os.path.exists( self._ssl_key_path )
        
        if cert_here ^ key_here:
            
            raise Exception( 'While creating the server database, only one of the paths "{}" and "{}" existed. You can create a db with these files already in place, but please either delete the existing file (to have hydrus generate its own pair) or find the other in the pair (to use your own).'.format( self._ssl_cert_path, self._ssl_key_path ) )
            
        elif not ( cert_here or key_here ):
            
            HydrusData.Print( 'Generating new cert/key files.' )
            
            if not HydrusEncryption.CRYPTO_OK:
                
                raise Exception( 'The database was asked for ssl cert and keys to start either the server or the client api in https. The files do not exist yet, so the database wanted to create new ones, but unfortunately "cryptography" library is not available, so this cannot be done. If you are running from source, please install this module using pip. Or drop in your own client.crt/client.key or server.crt/server.key files in the db directory.' )
                
            
            HydrusEncryption.GenerateOpenSSLCertAndKeyFile( self._ssl_cert_path, self._ssl_key_path )
            
        
        return ( self._ssl_cert_path, self._ssl_key_path )
        
    
    def GetStatus( self ):
        
        return ( self._current_status, self._current_job_name )
        
    
    def IsConnected( self ):
        
        return self._is_connected
        
    
    def IsDBUpdated( self ):
        
        return self._is_db_updated
        
    
    def IsFirstStart( self ):
        
        return self._is_first_start
        
    
    def LoopIsFinished( self ):
        
        return self._loop_finished
        
    
    def JobsQueueEmpty( self ):
        
        return self._jobs.empty()
        
    
    def MainLoop( self ):
        
        try:
            
            self._InitDBConnection() # have to reinitialise because the thread id has changed
            
            self._InitCaches()
            
        except:
            
            self._DisplayCatastrophicError( traceback.format_exc() )
            
            self._could_not_initialise = True
            
            return
            
        
        self._ready_to_serve_requests = True
        
        error_count = 0
        
        while not ( ( self._local_shutdown or HG.model_shutdown ) and self._jobs.empty() ):
            
            try:
                
                job = self._jobs.get( timeout = 1 )
                
                self._currently_doing_job = True
                self._current_job_name = job.ToString()
                
                self.publish_status_update()
                
                try:
                    
                    if HG.db_report_mode:
                        
                        summary = 'Running db job: ' + job.ToString()
                        
                        HydrusData.ShowText( summary )
                        
                    
                    if HG.profile_mode:
                        
                        summary = 'Profiling db job: ' + job.ToString()
                        
                        HydrusProfiling.Profile( summary, 'self._ProcessJob( job )', globals(), locals(), min_duration_ms = HG.db_profile_min_job_time_ms )
                        
                    else:
                        
                        self._ProcessJob( job )
                        
                    
                    error_count = 0
                    
                except:
                    
                    error_count += 1
                    
                    if error_count > 5:
                        
                        raise
                        
                    
                    self._jobs.put( job ) # couldn't lock db; put job back on queue
                    
                    time.sleep( 5 )
                    
                
                self._currently_doing_job = False
                self._current_job_name = ''
                
                self._finished_job_event.set()
                
                self.publish_status_update()
                
            except queue.Empty:
                
                if self._cursor_transaction_wrapper.TimeToCommit():
                    
                    self._cursor_transaction_wrapper.CommitAndBegin()
                    
                
            
            if self._pause_and_disconnect:
                
                self._CloseDBConnection()
                
                while self._pause_and_disconnect:
                    
                    if self._local_shutdown or HG.model_shutdown:
                        
                        break
                        
                    
                    time.sleep( 1 )
                    
                
                self._InitDBConnection()
                
            
        
        self._CloseDBConnection()
        
        temp_path = os.path.join( self._db_dir, self._durable_temp_db_filename )
        
        HydrusPaths.DeletePath( temp_path )
        
        self._loop_finished = True
        
    
    def PauseAndDisconnect( self, pause_and_disconnect ):
        
        self._pause_and_disconnect = pause_and_disconnect
        
    
    def Read( self, action, *args, **kwargs ):
        
        if action in self.READ_WRITE_ACTIONS:
            
            job_type = 'read_write'
            
        else:
            
            job_type = 'read'
            
        
        synchronous = True
        
        job = self._GenerateDBJob( job_type, synchronous, action, *args, **kwargs )
        
        if HG.model_shutdown:
            
            raise HydrusExceptions.ShutdownException( 'Application has shut down!' )
            
        
        self._jobs.put( job )
        
        return job.GetResult()
        
    
    def ReadyToServeRequests( self ):
        
        return self._ready_to_serve_requests
        
    
    def Shutdown( self ):
        
        self._local_shutdown = True
        
    
    def WaitUntilFree( self ):
        
        while True:
            
            if HG.model_shutdown:
                
                raise HydrusExceptions.ShutdownException( 'Application shutting down!' )
                
            elif self.JobsQueueEmpty() and not self.CurrentlyDoingJob():
                
                return
                
            else:
                
                self._finished_job_event.wait( 0.5 )
                
                self._finished_job_event.clear()
                
            
        
    
    def Write( self, action, synchronous, *args, **kwargs ):
        
        job_type = 'write'
        
        job = self._GenerateDBJob( job_type, synchronous, action, *args, **kwargs )
        
        if HG.model_shutdown:
            
            raise HydrusExceptions.ShutdownException( 'Application has shut down!' )
            
        
        self._jobs.put( job )
        
        if synchronous: return job.GetResult()
        
    
