"use strict";
Grabber.makeArray = function (val, allowFalsy) {
    if (allowFalsy === void 0) { allowFalsy = false; }
    if (!val && !allowFalsy) {
        return [];
    }
    if (!Array.isArray(val)) {
        return [val];
    }
    return val;
};
Grabber.regexMatch = function (regexp, src) {
    var matches = Grabber.regexMatches(regexp, src);
    if (matches && matches.length > 0) {
        return matches[0];
    }
    return undefined;
};
Grabber.mapObject = function (obj, fn) {
    var ret = {};
    for (var k in obj) {
        ret[k] = fn(obj[k]);
    }
    return ret;
};
Grabber.typedXML = function (val) {
    if (val && typeof val === "object" && ("#text" in val || "@attributes" in val)) {
        var txt = val["#text"];
        var isNil = "@attributes" in val && "nil" in val["@attributes"] && val["@attributes"]["nil"] === "true";
        if (isNil) {
            return null;
        }
        var type = "@attributes" in val && "type" in val["@attributes"] ? val["@attributes"]["type"] : undefined;
        if (type === "integer") {
            return parseInt(txt, 10);
        }
        else if (type === "array") {
            delete val["@attributes"]["type"];
            if (Object.keys(val["@attributes"]).length === 0) {
                delete val["@attributes"];
            }
            return Grabber.mapObject(val, Grabber.typedXML);
        }
        if (txt !== undefined) {
            return txt;
        }
    }
    if (val && val instanceof Array) {
        return val.map(Grabber.typedXML);
    }
    if (val && typeof val === "object") {
        if (Object.keys(val).length === 0) {
            return "";
        }
        return Grabber.mapObject(val, Grabber.typedXML);
    }
    return val;
};
/**
 * Set a value in an object using the dot ("a.b.c") path notation.
 */
function _set(obj, path, value) {
    var parts = path.split(".");
    for (var i = 0; i < parts.length - 1; ++i) {
        var part = parts[i];
        if (!(part in obj)) {
            obj[part] = {};
        }
        obj = obj[part];
    }
    obj[parts[parts.length - 1]] = value;
}
/**
 * Get a value in an object using the dot ("a.b.c") path notation.
 */
function _get(obj, path) {
    return path.split(".").reduce(function (ctx, part) { return ctx === null || ctx === void 0 ? void 0 : ctx[part]; }, obj);
}
Grabber.mapFields = function (data, map) {
    var result = {};
    if (typeof data !== "object") {
        return result;
    }
    for (var to in map) {
        var from = map[to];
        var value = _get(data, from);
        if (value !== null) {
            _set(result, to, value);
        }
    }
    return result;
};
Grabber.countToInt = function (str) {
    if (!str) {
        return undefined;
    }
    var count;
    var normalized = str.toLowerCase().trim().replace(/,/g, "");
    if (normalized.slice(-1) === "k") {
        var withoutK = normalized.substring(0, normalized.length - 1).trim();
        count = parseFloat(withoutK) * 1000;
    }
    else {
        count = parseFloat(normalized);
    }
    return Math.round(count);
};
Grabber.fileSizeToInt = function (str) {
    if (typeof str !== "string") {
        return str;
    }
    var res = str.match(/^(\d+(?:\.\d+)?)\s*(\w+)$/);
    if (res) {
        var val = parseFloat(res[1]);
        var unit = res[2].toLowerCase();
        if (unit === "mb") {
            return Math.round(val * 1024 * 1024);
        }
        if (unit === "kb") {
            return Math.round(val * 1024);
        }
        return Math.round(val);
    }
    return parseInt(str, 10);
};
Grabber.fixPageUrl = function (url, page, previous, pageTransformer) {
    if (!pageTransformer) {
        pageTransformer = function (p) { return p; };
    }
    url = url.replace("{page}", String(pageTransformer(page)));
    if (previous) {
        url = url.replace("{min}", previous.minId);
        url = url.replace("{max}", previous.maxId);
        url = url.replace("{min-1}", previous.minIdM1);
        url = url.replace("{max+1}", previous.maxIdP1);
    }
    return url;
};
Grabber.pageUrl = function (page, previous, limit, ifBelow, ifPrev, ifNext, pageTransformer) {
    var pageLimit = pageTransformer ? pageTransformer(page) : page;
    if (pageLimit <= limit || limit < 0) {
        return Grabber.fixPageUrl(ifBelow, page, previous, pageTransformer);
    }
    if (previous && previous.page === page + 1) {
        return Grabber.fixPageUrl(ifPrev, page, previous, pageTransformer);
    }
    if (previous && previous.page === page - 1) {
        return Grabber.fixPageUrl(ifNext, page, previous, pageTransformer);
    }
    throw new Error("You need valid previous page information to browse that far");
};
Grabber.regexToImages = function (regexp, src) {
    var images = [];
    var matches = Grabber.regexMatches(regexp, src);
    for (var _i = 0, matches_1 = matches; _i < matches_1.length; _i++) {
        var match = matches_1[_i];
        if ("json" in match) {
            var json = JSON.parse(match["json"]);
            for (var key in json) {
                match[key] = json[key];
            }
        }
        if (match.id) {
            match.id = parseInt(match.id, 10);
        }
        if (match.file_size) {
            match.file_size = Grabber.fileSizeToInt(match.file_size);
        }
        images.push(match);
    }
    return images;
};
Grabber.pick = function (obj, keys) {
    return keys.reduce(function (ret, key) {
        if (key in obj && obj[key] !== undefined) {
            ret[key] = obj[key];
        }
        return ret;
    }, {});
};
Grabber.regexToTags = function (regexp, src) {
    var tags = [];
    var uniques = {};
    var matches = Grabber.regexMatches(regexp, src);
    for (var _i = 0, matches_2 = matches; _i < matches_2.length; _i++) {
        var match = matches_2[_i];
        if (match["name"] in uniques) {
            continue;
        }
        if ("count" in match) {
            match["count"] = Grabber.countToInt(match["count"]);
        }
        tags.push(Grabber.pick(match, ["id", "name", "count", "type", "typeId"]));
        uniques[match["name"]] = true;
    }
    return tags;
};
Grabber.regexToPools = function (regexp, src) {
    var pools = [];
    var matches = Grabber.regexMatches(regexp, src);
    for (var _i = 0, matches_3 = matches; _i < matches_3.length; _i++) {
        var match = matches_3[_i];
        pools.push(match);
    }
    return pools;
};
Grabber.regexToConst = function (key, regexp, src) {
    var match = Grabber.regexMatch(regexp, src);
    if (match) {
        return match[key];
    }
    return undefined;
};
function _visitSearch(search, tag, and, or) {
    if ("operator" in search) {
        var left = _visitSearch(search.left, tag, and, or);
        var right = _visitSearch(search.right, tag, and, or);
        var func = search.operator === "and" ? and : or;
        return func(left, right);
    }
    else {
        return tag(search);
    }
}
Grabber.visitSearch = _visitSearch;
Grabber.buildQueryParams = function (params) {
    var ret = [];
    for (var key in params) {
        ret.push(encodeURIComponent(key) + "=" + encodeURIComponent(params[key]));
    }
    return ret.join("&");
};
function parseSearchVal(value, meta) {
    if (meta.type === "options") {
        for (var _i = 0, _a = meta.options; _i < _a.length; _i++) {
            var opt = _a[_i];
            var optVal = typeof opt === "string" ? opt : opt.value;
            if (optVal === value) {
                return value;
            }
        }
    }
    else if (meta.type === "input") {
        return meta.parser ? meta.parser(value) : value;
    }
    else if (meta.type === "bool") {
        if (value === "true" || value === "yes" || value === "1") {
            return true;
        }
        else if (value === "false" || value === "no" || value === "0") {
            return false;
        }
        console.warn("Unknown boolean value: \"".concat(value, "\""));
        return Boolean(value);
    }
    return null;
}
Grabber.parseSearchQuery = function (query, metas) {
    var tags = [];
    var ret = {};
    // Default values
    for (var key in metas) {
        if (metas[key].default) {
            ret[key] = metas[key].default;
        }
    }
    // Parse search
    for (var _i = 0, _a = query.split(" "); _i < _a.length; _i++) {
        var part = _a[_i];
        var index = part.indexOf(":");
        if (index !== -1) {
            var key = part.substring(0, index);
            if (key in metas) {
                var val = parseSearchVal(part.substring(index + 1), metas[key]);
                if (val !== null) {
                    ret[key] = val;
                    continue;
                }
            }
        }
        tags.push(part);
    }
    ret.query = tags.join(" ");
    return ret;
};
// Fix console calls since C++ handlers can't get variadic arguments
var originalConsole = console;
function argToString(arg) {
    if (typeof arg === "object") {
        return JSON.stringify(arg);
    }
    return String(arg);
}
function argsToString(args) {
    return args.map(argToString).join(' ');
}
// @ts-ignore
console = {
    debug: function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return originalConsole.debug(argsToString(args));
    },
    error: function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return originalConsole.error(argsToString(args));
    },
    info: function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return originalConsole.info(argsToString(args));
    },
    log: function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return originalConsole.log(argsToString(args));
    },
    warn: function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return originalConsole.warn(argsToString(args));
    },
};
