/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.impldep.org.junit.platform.commons.util;

import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.gradle.internal.impldep.org.junit.platform.commons.PreconditionViolationException;
import org.gradle.internal.impldep.org.junit.platform.commons.function.Try;
import org.gradle.internal.impldep.org.junit.platform.commons.logging.Logger;
import org.gradle.internal.impldep.org.junit.platform.commons.logging.LoggerFactory;
import org.gradle.internal.impldep.org.junit.platform.commons.util.ClassFileVisitor;
import org.gradle.internal.impldep.org.junit.platform.commons.util.ClassFilter;
import org.gradle.internal.impldep.org.junit.platform.commons.util.CloseablePath;
import org.gradle.internal.impldep.org.junit.platform.commons.util.Preconditions;
import org.gradle.internal.impldep.org.junit.platform.commons.util.StringUtils;
import org.gradle.internal.impldep.org.junit.platform.commons.util.UnrecoverableExceptions;

class ClasspathScanner {
    private static final Logger logger = LoggerFactory.getLogger(ClasspathScanner.class);
    private static final char CLASSPATH_RESOURCE_PATH_SEPARATOR = '/';
    private static final String CLASSPATH_RESOURCE_PATH_SEPARATOR_STRING = String.valueOf('/');
    private static final char PACKAGE_SEPARATOR_CHAR = '.';
    private static final String PACKAGE_SEPARATOR_STRING = String.valueOf('.');
    private static final String MALFORMED_CLASS_NAME_ERROR_MESSAGE = "Malformed class name";
    private final Supplier<ClassLoader> classLoaderSupplier;
    private final BiFunction<String, ClassLoader, Try<Class<?>>> loadClass;

    ClasspathScanner(Supplier<ClassLoader> classLoaderSupplier, BiFunction<String, ClassLoader, Try<Class<?>>> loadClass) {
        this.classLoaderSupplier = classLoaderSupplier;
        this.loadClass = loadClass;
    }

    List<Class<?>> scanForClassesInPackage(String basePackageName, ClassFilter classFilter) {
        Preconditions.condition("".equals(basePackageName) || StringUtils.isNotBlank(basePackageName), "basePackageName must not be null or blank");
        Preconditions.notNull(classFilter, "classFilter must not be null");
        basePackageName = basePackageName.trim();
        return this.findClassesForUris(this.getRootUrisForPackageNameOnClassPathAndModulePath(basePackageName), basePackageName, classFilter);
    }

    List<Class<?>> scanForClassesInClasspathRoot(URI root, ClassFilter classFilter) {
        Preconditions.notNull(root, "root must not be null");
        Preconditions.notNull(classFilter, "classFilter must not be null");
        return this.findClassesForUri(root, "", classFilter);
    }

    private List<Class<?>> findClassesForUris(List<URI> baseUris, String basePackageName, ClassFilter classFilter) {
        return baseUris.stream().map(baseUri -> this.findClassesForUri((URI)baseUri, basePackageName, classFilter)).flatMap(Collection::stream).distinct().collect(Collectors.toList());
    }

    private List<Class<?>> findClassesForUri(URI baseUri, String basePackageName, ClassFilter classFilter) {
        List<Class<?>> list;
        block9: {
            CloseablePath closeablePath = CloseablePath.create(baseUri);
            try {
                Path baseDir = closeablePath.getPath();
                list = this.findClassesForPath(baseDir, basePackageName, classFilter);
                if (closeablePath == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (closeablePath != null) {
                        try {
                            closeablePath.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (PreconditionViolationException ex) {
                    throw ex;
                }
                catch (Exception ex) {
                    logger.warn(ex, () -> "Error scanning files for URI " + baseUri);
                    return Collections.emptyList();
                }
            }
            closeablePath.close();
        }
        return list;
    }

    private List<Class<?>> findClassesForPath(Path baseDir, String basePackageName, ClassFilter classFilter) {
        Preconditions.condition(Files.exists(baseDir, new LinkOption[0]), () -> "baseDir must exist: " + baseDir);
        ArrayList classes = new ArrayList();
        try {
            Files.walkFileTree(baseDir, new ClassFileVisitor(classFile -> this.processClassFileSafely(baseDir, basePackageName, classFilter, (Path)classFile, classes::add)));
        }
        catch (IOException ex) {
            logger.warn(ex, () -> "I/O error scanning files in " + baseDir);
        }
        return classes;
    }

    private void processClassFileSafely(Path baseDir, String basePackageName, ClassFilter classFilter, Path classFile, Consumer<Class<?>> classConsumer) {
        try {
            String fullyQualifiedClassName = this.determineFullyQualifiedClassName(baseDir, basePackageName, classFile);
            if (classFilter.match(fullyQualifiedClassName)) {
                try {
                    this.loadClass.apply(fullyQualifiedClassName, this.getClassLoader()).toOptional().filter(classFilter).ifPresent(classConsumer);
                }
                catch (InternalError internalError) {
                    this.handleInternalError(classFile, fullyQualifiedClassName, internalError);
                }
            }
        }
        catch (Throwable throwable) {
            this.handleThrowable(classFile, throwable);
        }
    }

    private String determineFullyQualifiedClassName(Path baseDir, String basePackageName, Path classFile) {
        return Stream.of(basePackageName, this.determineSubpackageName(baseDir, classFile), this.determineSimpleClassName(classFile)).filter(value -> !value.isEmpty()).collect(Collectors.joining(PACKAGE_SEPARATOR_STRING));
    }

    private String determineSimpleClassName(Path classFile) {
        String fileName = classFile.getFileName().toString();
        return fileName.substring(0, fileName.length() - ".class".length());
    }

    private String determineSubpackageName(Path baseDir, Path classFile) {
        Path relativePath = baseDir.relativize(classFile.getParent());
        String pathSeparator = baseDir.getFileSystem().getSeparator();
        String subpackageName = relativePath.toString().replace(pathSeparator, PACKAGE_SEPARATOR_STRING);
        if (subpackageName.endsWith(pathSeparator)) {
            subpackageName = subpackageName.substring(0, subpackageName.length() - pathSeparator.length());
        }
        return subpackageName;
    }

    private void handleInternalError(Path classFile, String fullyQualifiedClassName, InternalError ex) {
        if (MALFORMED_CLASS_NAME_ERROR_MESSAGE.equals(ex.getMessage())) {
            this.logMalformedClassName(classFile, fullyQualifiedClassName, ex);
        } else {
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void handleThrowable(Path classFile, Throwable throwable) {
        UnrecoverableExceptions.rethrowIfUnrecoverable(throwable);
        this.logGenericFileProcessingException(classFile, throwable);
    }

    private void logMalformedClassName(Path classFile, String fullyQualifiedClassName, InternalError ex) {
        try {
            logger.debug(ex, () -> String.format("The java.lang.Class loaded from path [%s] has a malformed class name [%s].", classFile.toAbsolutePath(), fullyQualifiedClassName));
        }
        catch (Throwable t) {
            UnrecoverableExceptions.rethrowIfUnrecoverable(t);
            ex.addSuppressed(t);
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void logGenericFileProcessingException(Path classFile, Throwable throwable) {
        logger.debug(throwable, () -> String.format("Failed to load java.lang.Class for path [%s] during classpath scanning.", classFile.toAbsolutePath()));
    }

    private ClassLoader getClassLoader() {
        return this.classLoaderSupplier.get();
    }

    private List<URI> getRootUrisForPackageNameOnClassPathAndModulePath(String basePackageName) {
        LinkedHashSet<URI> uriSet = new LinkedHashSet<URI>(this.getRootUrisForPackage(basePackageName));
        if (!basePackageName.isEmpty() && !basePackageName.endsWith(PACKAGE_SEPARATOR_STRING)) {
            this.getRootUrisForPackage(basePackageName + PACKAGE_SEPARATOR_STRING).stream().map(ClasspathScanner::removeTrailingClasspathResourcePathSeparator).forEach(uriSet::add);
        }
        return new ArrayList<URI>(uriSet);
    }

    private static URI removeTrailingClasspathResourcePathSeparator(URI uri) {
        String string = uri.toString();
        if (string.endsWith(CLASSPATH_RESOURCE_PATH_SEPARATOR_STRING)) {
            return URI.create(string.substring(0, string.length() - 1));
        }
        return uri;
    }

    private static String packagePath(String packageName) {
        if (packageName.isEmpty()) {
            return "";
        }
        return packageName.replace('.', '/');
    }

    private List<URI> getRootUrisForPackage(String basePackageName) {
        try {
            Enumeration<URL> resources = this.getClassLoader().getResources(ClasspathScanner.packagePath(basePackageName));
            ArrayList<URI> uris = new ArrayList<URI>();
            while (resources.hasMoreElements()) {
                URL resource = resources.nextElement();
                uris.add(resource.toURI());
            }
            return uris;
        }
        catch (Exception ex) {
            logger.warn(ex, () -> "Error reading URIs from class loader for base package " + basePackageName);
            return Collections.emptyList();
        }
    }
}

