/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.storage.commitgraph;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import org.eclipse.jgit.annotations.NonNull;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.commitgraph.GraphCommits;
import org.eclipse.jgit.internal.storage.io.CancellableDigestOutputStream;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.util.NB;

public class CommitGraphWriter {
    private static final int COMMIT_GRAPH_VERSION_GENERATED = 1;
    private static final int OID_HASH_VERSION = 1;
    private static final int GRAPH_FANOUT_SIZE = 1024;
    private static final int GENERATION_NUMBER_MAX = 0x3FFFFFFF;
    private final int hashsz;
    private final GraphCommits graphCommits;

    public CommitGraphWriter(@NonNull GraphCommits graphCommits) {
        this.graphCommits = graphCommits;
        this.hashsz = 20;
    }

    public void write(@NonNull ProgressMonitor monitor2, @NonNull OutputStream commitGraphStream) throws IOException {
        if (this.graphCommits.size() == 0) {
            return;
        }
        List<ChunkHeader> chunks = this.createChunks();
        long writeCount = 256 + 2 * this.graphCommits.size() + this.graphCommits.getExtraEdgeCnt();
        monitor2.beginTask(MessageFormat.format(JGitText.get().writingOutCommitGraph, chunks.size()), (int)writeCount);
        try {
            try {
                Throwable throwable = null;
                Object var7_8 = null;
                try (CancellableDigestOutputStream out = new CancellableDigestOutputStream(monitor2, commitGraphStream);){
                    this.writeHeader(out, chunks.size());
                    this.writeChunkLookup(out, chunks);
                    this.writeChunks(monitor2, out, chunks);
                    this.writeCheckSum(out);
                }
                catch (Throwable throwable2) {
                    if (throwable == null) {
                        throwable = throwable2;
                    } else if (throwable != throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
            }
            catch (InterruptedIOException e) {
                throw new IOException(JGitText.get().commitGraphWritingCancelled, e);
            }
        }
        finally {
            monitor2.endTask();
        }
    }

    private List<ChunkHeader> createChunks() {
        ArrayList<ChunkHeader> chunks = new ArrayList<ChunkHeader>();
        chunks.add(new ChunkHeader(1330201670, 1024L));
        chunks.add(new ChunkHeader(1330201676, this.hashsz * this.graphCommits.size()));
        chunks.add(new ChunkHeader(1128546644, (this.hashsz + 16) * this.graphCommits.size()));
        if (this.graphCommits.getExtraEdgeCnt() > 0) {
            chunks.add(new ChunkHeader(0x45444745, this.graphCommits.getExtraEdgeCnt() * 4));
        }
        return chunks;
    }

    private void writeHeader(CancellableDigestOutputStream out, int numChunks) throws IOException {
        byte[] headerBuffer = new byte[8];
        NB.encodeInt32(headerBuffer, 0, 1128747080);
        byte[] byArray = new byte[4];
        byArray[0] = 1;
        byArray[1] = 1;
        byArray[2] = (byte)numChunks;
        byte[] buff = byArray;
        System.arraycopy(buff, 0, headerBuffer, 4, 4);
        out.write(headerBuffer, 0, 8);
        out.flush();
    }

    private void writeChunkLookup(CancellableDigestOutputStream out, List<ChunkHeader> chunks) throws IOException {
        int numChunks = chunks.size();
        long chunkOffset = 8 + (numChunks + 1) * 12;
        byte[] buffer = new byte[12];
        for (ChunkHeader chunk : chunks) {
            NB.encodeInt32(buffer, 0, chunk.id);
            NB.encodeInt64(buffer, 4, chunkOffset);
            out.write(buffer);
            chunkOffset += chunk.size;
        }
        NB.encodeInt32(buffer, 0, 0);
        NB.encodeInt64(buffer, 4, chunkOffset);
        out.write(buffer);
    }

    private void writeChunks(ProgressMonitor monitor2, CancellableDigestOutputStream out, List<ChunkHeader> chunks) throws IOException {
        for (ChunkHeader chunk : chunks) {
            int chunkId = chunk.id;
            switch (chunkId) {
                case 1330201670: {
                    this.writeFanoutTable(out);
                    break;
                }
                case 1330201676: {
                    this.writeOidLookUp(out);
                    break;
                }
                case 1128546644: {
                    this.writeCommitData(monitor2, out);
                    break;
                }
                case 0x45444745: {
                    this.writeExtraEdges(out);
                }
            }
        }
    }

    private void writeCheckSum(CancellableDigestOutputStream out) throws IOException {
        out.write(out.getDigest());
        out.flush();
    }

    private void writeFanoutTable(CancellableDigestOutputStream out) throws IOException {
        byte[] tmp = new byte[4];
        int[] fanout = new int[256];
        for (RevCommit c : this.graphCommits) {
            int n = c.getFirstByte() & 0xFF;
            fanout[n] = fanout[n] + 1;
        }
        int i = 1;
        while (i < fanout.length) {
            int n = i;
            fanout[n] = fanout[n] + fanout[i - 1];
            ++i;
        }
        int[] nArray = fanout;
        int n = fanout.length;
        int n2 = 0;
        while (n2 < n) {
            int n3 = nArray[n2];
            NB.encodeInt32(tmp, 0, n3);
            out.write(tmp, 0, 4);
            out.getWriteMonitor().update(1);
            ++n2;
        }
    }

    private void writeOidLookUp(CancellableDigestOutputStream out) throws IOException {
        byte[] tmp = new byte[4 + this.hashsz];
        for (RevCommit c : this.graphCommits) {
            c.copyRawTo(tmp, 0);
            out.write(tmp, 0, this.hashsz);
            out.getWriteMonitor().update(1);
        }
    }

    private void writeCommitData(ProgressMonitor monitor2, CancellableDigestOutputStream out) throws IOException {
        int[] generations = this.computeGenerationNumbers(monitor2);
        int num = 0;
        byte[] tmp = new byte[this.hashsz + 16];
        int i = 0;
        for (RevCommit commit : this.graphCommits) {
            RevCommit parent;
            int edgeValue;
            int[] packedDate = new int[2];
            RevTree treeId = commit.getTree();
            treeId.copyRawTo(tmp, 0);
            RevCommit[] parents = commit.getParents();
            if (parents.length == 0) {
                edgeValue = 0x70000000;
            } else {
                parent = parents[0];
                edgeValue = this.graphCommits.getOidPosition(parent);
            }
            NB.encodeInt32(tmp, this.hashsz, edgeValue);
            if (parents.length == 1) {
                edgeValue = 0x70000000;
            } else if (parents.length == 2) {
                parent = parents[1];
                edgeValue = this.graphCommits.getOidPosition(parent);
            } else if (parents.length > 2) {
                edgeValue = Integer.MIN_VALUE | num;
                num += parents.length - 1;
            }
            NB.encodeInt32(tmp, this.hashsz + 4, edgeValue);
            packedDate[0] = 0;
            packedDate[0] = packedDate[0] | generations[i] << 2;
            packedDate[1] = commit.getCommitTime();
            NB.encodeInt32(tmp, this.hashsz + 8, packedDate[0]);
            NB.encodeInt32(tmp, this.hashsz + 12, packedDate[1]);
            out.write(tmp);
            out.getWriteMonitor().update(1);
            ++i;
        }
    }

    private int[] computeGenerationNumbers(ProgressMonitor monitor2) throws MissingObjectException {
        int[] generations = new int[this.graphCommits.size()];
        monitor2.beginTask(JGitText.get().computingCommitGeneration, this.graphCommits.size());
        for (RevCommit cmit : this.graphCommits) {
            monitor2.update(1);
            int generation = generations[this.graphCommits.getOidPosition(cmit)];
            if (generation != Constants.COMMIT_GENERATION_NOT_COMPUTED && generation != Constants.COMMIT_GENERATION_UNKNOWN) continue;
            Stack<RevCommit> commitStack = new Stack<RevCommit>();
            commitStack.push(cmit);
            while (!commitStack.empty()) {
                int maxGeneration = 0;
                boolean allParentComputed = true;
                RevCommit current = (RevCommit)commitStack.peek();
                int i = 0;
                while (i < current.getParentCount()) {
                    RevCommit parent = current.getParent(i);
                    generation = generations[this.graphCommits.getOidPosition(parent)];
                    if (generation == Constants.COMMIT_GENERATION_NOT_COMPUTED || generation == Constants.COMMIT_GENERATION_UNKNOWN) {
                        allParentComputed = false;
                        commitStack.push(parent);
                        break;
                    }
                    if (generation > maxGeneration) {
                        maxGeneration = generation;
                    }
                    ++i;
                }
                if (!allParentComputed) continue;
                RevCommit commit = (RevCommit)commitStack.pop();
                generation = maxGeneration + 1;
                if (generation > 0x3FFFFFFF) {
                    generation = 0x3FFFFFFF;
                }
                generations[this.graphCommits.getOidPosition((RevCommit)commit)] = generation;
            }
        }
        monitor2.endTask();
        return generations;
    }

    private void writeExtraEdges(CancellableDigestOutputStream out) throws IOException {
        byte[] tmp = new byte[4];
        for (RevCommit commit : this.graphCommits) {
            RevCommit[] parents = commit.getParents();
            if (parents.length <= 2) continue;
            int n = 1;
            while (n < parents.length) {
                RevCommit parent = parents[n];
                int edgeValue = this.graphCommits.getOidPosition(parent);
                if (n == parents.length - 1) {
                    edgeValue |= Integer.MIN_VALUE;
                }
                NB.encodeInt32(tmp, 0, edgeValue);
                out.write(tmp);
                out.getWriteMonitor().update(1);
                ++n;
            }
        }
    }

    private static class ChunkHeader {
        final int id;
        final long size;

        public ChunkHeader(int id, long size) {
            this.id = id;
            this.size = size;
        }
    }
}

