/*
 * Copyright 2021 Google LLC
 * SPDX-License-Identifier: MIT
 */

#ifndef VN_RING_H
#define VN_RING_H

#include "vn_common.h"

#include "vn_cs.h"

/**
 * A ring is a single-producer and single-consumer circular buffer.  The data
 * in the buffer are produced and consumed in order.  An externally-defined
 * mechanism is required for ring setup and notifications in both directions.
 * Notifications for new data from the producer are needed only when the
 * consumer is not actively polling, which is indicated by the ring status.
 *
 * For venus, the data are plain venus commands.  When a venus command is
 * consumed from the ring's perspective, there can still be ongoing CPU and/or
 * GPU works.  This is not an issue when the works generated by following
 * venus commands are correctly queued after the ongoing works.  There are
 * also venus commands that facilitate polling or waiting for ongoing works.
 */

/* the layout of a ring in a shmem */
struct vn_ring_layout {
   size_t head_offset;
   size_t tail_offset;
   size_t status_offset;

   size_t buffer_offset;
   size_t buffer_size;

   size_t extra_offset;
   size_t extra_size;

   size_t shmem_size;
};

void
vn_ring_get_layout(size_t buf_size,
                   size_t extra_size,
                   struct vn_ring_layout *layout);

struct vn_ring *
vn_ring_create(struct vn_instance *instance,
               const struct vn_ring_layout *layout,
               uint8_t direct_order);

void
vn_ring_destroy(struct vn_ring *ring);

uint64_t
vn_ring_get_id(struct vn_ring *ring);

uint32_t
vn_ring_load_status(const struct vn_ring *ring);

void
vn_ring_unset_status_bits(struct vn_ring *ring, uint32_t mask);

bool
vn_ring_get_seqno_status(struct vn_ring *ring, uint32_t seqno);

void
vn_ring_wait_all(struct vn_ring *ring);

struct vn_ring_submit_command {
   /* empty command implies errors */
   struct vn_cs_encoder command;
   struct vn_cs_encoder_buffer buffer;
   /* non-zero implies waiting */
   size_t reply_size;

   /* when reply_size is non-zero, NULL can be returned on errors */
   struct vn_renderer_shmem *reply_shmem;
   struct vn_cs_decoder reply;

   /* valid when ring submission succeeds */
   bool ring_seqno_valid;
   uint32_t ring_seqno;
};

static inline struct vn_cs_encoder *
vn_ring_submit_command_init(struct vn_ring *ring,
                            struct vn_ring_submit_command *submit,
                            void *cmd_data,
                            size_t cmd_size,
                            size_t reply_size)
{
   submit->buffer = VN_CS_ENCODER_BUFFER_INITIALIZER(cmd_data);
   submit->command = VN_CS_ENCODER_INITIALIZER(&submit->buffer, cmd_size);

   submit->reply_size = reply_size;
   submit->reply_shmem = NULL;

   submit->ring_seqno_valid = false;

   return &submit->command;
}

static inline struct vn_cs_decoder *
vn_ring_get_command_reply(struct vn_ring *ring,
                          struct vn_ring_submit_command *submit)
{
   return submit->reply_shmem ? &submit->reply : NULL;
}

void
vn_ring_free_command_reply(struct vn_ring *ring,
                           struct vn_ring_submit_command *submit);

void
vn_ring_submit_command(struct vn_ring *ring,
                       struct vn_ring_submit_command *submit);

VkResult
vn_ring_submit_command_simple(struct vn_ring *ring,
                              const struct vn_cs_encoder *cs);

#endif /* VN_RING_H */
