#include "i18n.h"

#include <QDir>
#include <QLocale>
#include <QApplication>

#include "initializableqmap.h"

const QString INT_LANG_DIR = ":/languages/";

I18N::I18N(QObject *parent) : QObject(parent)
{
}

QList<Language> I18N::getLanguages() {
    QList<Language> languages;

    QString defaultLocale = this->getDefaultLocale();

    QDir dir(INT_LANG_DIR);
    QStringList fileNames = dir.entryList(QStringList("*.qm"));

    for (int i = 0; i < fileNames.size(); ++i) {
        QString locale = fileNames[i];
        locale.truncate(locale.lastIndexOf('.'));

        QString lang = QLocale::languageToString(QLocale(locale).language());

        Language language;
        language.iconFilePath = QString("%1%2.png").arg(INT_LANG_DIR).arg(locale);
        language.locale = locale;
        language.isDefault = defaultLocale == locale;
        language.name = lang;
        languages.append(language);
    }

    return languages;
}

QString I18N::getDefaultLocale() {
    QString defaultLocale = QLocale::system().name();
    defaultLocale.truncate(defaultLocale.lastIndexOf('_'));
    return defaultLocale;
}

void I18N::loadLanguage(const QString& language) {
    if (mCurrentLanguage != language) {
        mCurrentLanguage = language;
        QLocale locale = QLocale(mCurrentLanguage);
        QLocale::setDefault(locale);
        switchTranslator(mTranslator, QString("%1.qm").arg(language));
    }
}

void I18N::switchTranslator(QTranslator& translator, const QString& filename) {
    qApp->removeTranslator(&translator);

    if (translator.load(filename, INT_LANG_DIR)) qApp->installTranslator(&translator);
}

void I18N::onSystemLocalChanged() {
    this->loadLanguage(this->getDefaultLocale());
}

const InitializableQMap<QString, QString> GOOGLE_PLAY_STORE_IMAGES = InitializableQMap<QString, QString>()
<< QPair<QString, QString>("en", "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")
<< QPair<QString, QString>("de", "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")
<< QPair<QString, QString>("cs", "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");

QString I18N::getBase64GooglePlayStoreIconForCurrentLanguage() {
    return GOOGLE_PLAY_STORE_IMAGES.value(mCurrentLanguage);
}
