const std = @import("../std.zig");
const assert = std.debug.assert;
const mem = std.mem;

context: *const anyopaque,
writeFn: *const fn (context: *const anyopaque, bytes: []const u8) anyerror!usize,

const Self = @This();
pub const Error = anyerror;

pub fn write(self: Self, bytes: []const u8) anyerror!usize {
    return self.writeFn(self.context, bytes);
}

pub fn writeAll(self: Self, bytes: []const u8) anyerror!void {
    var index: usize = 0;
    while (index != bytes.len) {
        index += try self.write(bytes[index..]);
    }
}

pub fn print(self: Self, comptime format: []const u8, args: anytype) anyerror!void {
    return std.fmt.format(self, format, args);
}

pub fn writeByte(self: Self, byte: u8) anyerror!void {
    const array = [1]u8{byte};
    return self.writeAll(&array);
}

pub fn writeByteNTimes(self: Self, byte: u8, n: usize) anyerror!void {
    var bytes: [256]u8 = undefined;
    @memset(bytes[0..], byte);

    var remaining: usize = n;
    while (remaining > 0) {
        const to_write = @min(remaining, bytes.len);
        try self.writeAll(bytes[0..to_write]);
        remaining -= to_write;
    }
}

pub fn writeBytesNTimes(self: Self, bytes: []const u8, n: usize) anyerror!void {
    var i: usize = 0;
    while (i < n) : (i += 1) {
        try self.writeAll(bytes);
    }
}

pub inline fn writeInt(self: Self, comptime T: type, value: T, endian: std.builtin.Endian) anyerror!void {
    var bytes: [@divExact(@typeInfo(T).Int.bits, 8)]u8 = undefined;
    mem.writeInt(std.math.ByteAlignedInt(@TypeOf(value)), &bytes, value, endian);
    return self.writeAll(&bytes);
}

pub fn writeStruct(self: Self, value: anytype) anyerror!void {
    // Only extern and packed structs have defined in-memory layout.
    comptime assert(@typeInfo(@TypeOf(value)).Struct.layout != .Auto);
    return self.writeAll(mem.asBytes(&value));
}
